/*****************************************************************************
 *
 * Project:       
 *
 * Module:		Memory Management Module( For short MMM )
 *
 * Copyright:	(c) Powermatic Data Systems Ltd.
 *              
 *
 * $Author:  	Shao Guohua $
 * $Revision:   1.0  $
 * $Date:    08 Jan 2003  $

******************************************************************************
1) Physical Memory figure( Low Address  -------------> High Address ) 

 Store            Store all Blks    <--Smallest Block----------><--Bigger Block---------><--Biggest Block------>   
 BlkType's        Entry of one         all blocks of              all blocks of            all blocks of 
 Entry            specific BlkType   First mmmBlkTypeEntry      Sec mmmBlkTypeEntry      ast mmmBlkTypeEntry
 (First Free,
  Then used ) 
_________________________________________________________________________________________________________ 
|mmmBlkTypeEntry  |   mmmBlkEntry   |Blk 1 | Blk 2 |........ |Blk last |Blk 1|Blk 2|...|Blk last|...|Blk 1| Blk 2|...|Blk last|
--------------------------------------------------------------------------------------------------------- 
PhyAddr_TypeEntry  PhyAddr_BlkEntry  PhyAddr_Block
PhyAddr_mmm


2) MMM List Figure:
    
           (Node)                   (Node)                    (Node)
mmmVar.FreeList-->First mmmBlkTypeEntry--->Sec mmmBlkTypeEntry...->...-->last mmmBlkTypeEntry
                 (smallest block)       ( bigger block)            ( biggest) 
                    |                       |                         |
                    |(LIST)                 |(LIST)                   |(LIST)
                    |                       |                         |
                pmemEntryList          pmemEntryList              pmemEntryList
                    |                       |                         |
                    |(Node)                 |(Node)                   |(Node)
                    |                       |                         |
                First mmmBlkEntry    First mmmBlkEntry        First mmmBlkEntry                    
                    |                       |                         |
                    |(Node)                 |(Node)                   |(Node)
                    |                       |                         |
                sec mmmBlkEntry      sec mmmBlkEntry          Sec mmmBlkEntry
                    |                       |                         |
                    |(Node)                 |(Node)                   |(Node)
                    .                       .                         .
                    .                       .                         .
                    |                       |                         |
                    |                       |                         |
                Last mmmBlkEntry      Last mmmBlkEntry         Last mmmBlkEntry
                
Note, mmmUsedFreeList's figure is same to mmmVar.FreeList.

******************************************************************************
Note, 
  1) if define NEED_MMM_SUPPORT  1, then it will use memory mangament module,
    if will avoid memory scrape block
  2) if define NEED_MMM_SUPPORT   0, then it will use malloc/free/realloc in fact, ie, not use mmm.

*****************************************************************************/
#include "vxworks.h"
#include "stdarg.h"
#include "semlib.h"
#include "stdlib.h"
#include "ticklib.h"
#include "taskLib.h"
#include "lstLib.h"
#include "stdio.h"
#include "stdlib.h"
#include "string.h"
#include "sysLib.h"

#include "MemMgrm.h"


/************************************************
Suggestion configuration for release version:

#define MMM_RECORD_MODULENAME	    1  
#define MMM_RECORD_TIME             0  
#define MMM_RECORD_TASK		        0  
#define MMM_RECORD_REQUEST_SIZE  	0  


#define MMM_DEBUG                   0
************************************************/


#define MMM_RECORD_MODULENAME	    1  /* 0--not record the applicant'smodulename, 1--record the applicant'smodulename*/
#define MMM_RECORD_TIME             1  /* 0--not record the allocation time, 1--record the allocation time*/
#define MMM_RECORD_TASK		        0  /* 0--not record the applicant's task, 1--record the applicant's task*/

#define MMM_RECORD_REQUEST_SIZE  	0   /*0--not record request size, 1--record request size*/

#define MMM_DEBUG                   0 /*1-open debug, 0-close debug */
#define MMM_DEBUG1					1 /*1-open debug, 0-close debug */


#define NEED_MMM_SUPPORT 			1 /*0--not use mmm, but vxworks  1-use mmm*/


#if NEED_MMM_SUPPORT

	#define MMM_MAX_MODULENAME_LEN       (8)  /*max length of the module name:mmm_MAX_MODULENAME_LEN -1  */
	#define MMM_MAX_BLOCK_TYPES          30   /* max block size type */
	#define MMM_MEMALIGN_BYTES        (4)
	
	//list structure: each one means a kind of block with specific size
	typedef struct mmmBlkTypeEntry
	{
		NODE   	Node;    		/*for mmmBlkTypeEntry LIST */
		LIST    pmemEntryList;
		size_t  nBlockSize;
		size_t  nBlockNum;
		char*   PhyAddr_Begin;
		char*   PhyAddr_End;
		BOOL    bfFixed;

		ULONG ulRequestTimes;
		ULONG ulRespondOKTimes;
		ULONG ulRespondBadTimes;
		ULONG ulRespondFailTimes;

	} mmmBlkTypeEntry;

	//memory structure to keep one block's information
	typedef struct mmmBlkEntry
	{
		NODE   	Node;    	/*for mmmBlkEntry LIST */
		void*   BlockAddr;

		#if MMM_RECORD_MODULENAME	
			char 	ModuleName[MMM_MAX_MODULENAME_LEN]; //the owner of the entry
		#endif	

		#if MMM_RECORD_TIME
			ULONG 	ulTicks;  //the begining ticks of the owner begin to use
		#endif

		#if MMM_RECORD_TASK
			int     taskID;  //the taskID of the owner		
		#endif
			
		#if MMM_RECORD_REQUEST_SIZE		
			int     nReqBlkSize; 
		#endif		
	
	} mmmBlkEntry;

	typedef struct mmm_fit_schem
	{
		size_t nBlockSize;
		size_t nBlockNum;
		BOOL bfFixed;
	}mmm_fit_schem;

	typedef struct mmm_Variant
	{
		LIST FreeList;  
		LIST UsedList;
		
		SEM_ID mmmSemID;
		BOOL   mmmOK;

		//physical memory address
		char *PhyAddr_mmm;
		char *PhyAddr_TypeEntry;
		char *PhyAddr_BlkEntry;
		char *PhyAddr_Block;

		//Store fit memory scheme
		int nDraftSchemNum, nFitSchemNum;
		mmm_fit_schem mmmDraftSchem[MMM_MAX_BLOCK_TYPES];
		mmm_fit_schem mmmFitSchem[MMM_MAX_BLOCK_TYPES];
		USHORT sysTickRate;
		
	}mmm_Variant;

	LOCAL mmm_Variant mmmVar={0};	

	LOCAL int GetBlockSize_private(void *pBlock);
    LOCAL int GetListNo_private(void *pBlock);
	LOCAL mmmBlkEntry *mmm_GetFreeEntry( size_t nBytes, char *ModuleName );
	LOCAL BOOL InitMmmList(mmm_fit_schem pFitSchem[MMM_MAX_BLOCK_TYPES], int nBlockTypes);
	LOCAL void PrintHeader();
	LOCAL int mmmComparQSort( mmm_fit_schem *First, mmm_fit_schem *Second);
	
	BOOL mmm_init(MMM_CFG *pmmmCfg)
	{
		//size_t nBlockSize=0, nBlockNum=0;
		//va_list vargs;
		int  nAllFitBlock = 0;
		//BOOL tmpFixed;
		
	#if MMM_DEBUG1
		int  i ;
	#endif

		mmmVar.sysTickRate = sysClkRateGet();

		if( pmmmCfg == NULL ) 
		{
			printf("mmm_init paramenter error!\r\n");
			return FALSE;			
		}					

		nAllFitBlock = 0;
			
		/*Get draft memeory schem from vargs*/		
		do
		{		
			if( pmmmCfg->nBlockSize == 0 ) break;
			if( pmmmCfg->nBlockNum == 0) goto LOOP_NEXT;

			mmmVar.mmmDraftSchem[nAllFitBlock].nBlockSize = pmmmCfg->nBlockSize;
			mmmVar.mmmDraftSchem[nAllFitBlock].nBlockNum = pmmmCfg->nBlockNum;
			mmmVar.mmmDraftSchem[nAllFitBlock].bfFixed = pmmmCfg->bfFixed;

			if( ( mmmVar.mmmDraftSchem[nAllFitBlock].nBlockSize % MMM_MEMALIGN_BYTES ) != 0 )
				mmmVar.mmmDraftSchem[nAllFitBlock].nBlockSize = 
			  		( mmmVar.mmmDraftSchem[nAllFitBlock].nBlockSize / MMM_MEMALIGN_BYTES + 1) * MMM_MEMALIGN_BYTES;
			
			nAllFitBlock ++;

			if( nAllFitBlock >= MMM_MAX_BLOCK_TYPES ) break;

			LOOP_NEXT: pmmmCfg++;
			
		} while(1);

		mmmVar.nDraftSchemNum = nAllFitBlock;		

		if( nAllFitBlock == 0) return FALSE; //No non-zero block

		#if MMM_DEBUG1
			printf("Draft memory schem: \r\n" );
			for(i =0; i<nAllFitBlock; i++ )
				printf("Block %02d: size %d, num %d (%s)\r\n", i, mmmVar.mmmDraftSchem[i].nBlockSize, mmmVar.mmmDraftSchem[i].nBlockNum, 
				(mmmVar.mmmDraftSchem[i].bfFixed) ? "Fixed":"Not Fixed" );
		#endif


		/*Make Fit Scheme */
		while(1)
		{
			memcpy( mmmVar.mmmFitSchem, mmmVar.mmmDraftSchem, nAllFitBlock * sizeof(mmm_fit_schem));
			mmmVar.nFitSchemNum = nAllFitBlock;

			qsort( mmmVar.mmmFitSchem, mmmVar.nFitSchemNum, sizeof(mmm_fit_schem), (FUNCPTR)mmmComparQSort );
			
			if( InitMmmList( mmmVar.mmmFitSchem, mmmVar.nFitSchemNum ) )
			{				
				break;
			}

			//if fail, then delete last and try to do again.
			nAllFitBlock --; //delete last one 

			if( nAllFitBlock == 0 ) 
			{
				printf("\r\nThere is no enough memory to support Memory Managemet Module\r\n");
				printf("Please check your mmm's configurations.\r\n");
				return FALSE;
			}	
			
		}

		#if MMM_DEBUG1
			printf("\r\nFit memory schem: \r\n" );
			for(i =0; i<mmmVar.nFitSchemNum; i++ )
				printf("No. %d--%d %d(%s).\r\n", i, mmmVar.mmmFitSchem[i].nBlockSize, mmmVar.mmmFitSchem[i].nBlockNum,
				(mmmVar.mmmFitSchem[i].bfFixed) ? "Fixed":"Not Fixed" );

			if( nAllFitBlock != mmmVar.nDraftSchemNum )
				printf("\r\nMemory Management Module's configuration has been adjusted.\r\n");
		#endif
		
		
		return TRUE;		
	}


	/*******************************************
		Suppose a fit memory schem is put in the mmmFitSchem[],
		Before call InitMmmList, must check mmmFitSchem[], all its number must be non-Zero,
  	    Otherwise it will fail later.
    *******************************************/
	BOOL InitMmmList(mmm_fit_schem pFitSchem[MMM_MAX_BLOCK_TYPES], int nBlockTypes)
	{
		int i, j;		
		int nAllBlockNum  = 0;
		size_t ulNeedMaxPhySize = 0;
		char *pListEntryPhyAddr = NULL;
		char *pMemEntryPhyAddr = NULL;
		char *pBlockAddr = NULL;
		mmmBlkTypeEntry* pFreeBlkTypeList = NULL;
		mmmBlkTypeEntry* pUsedBlkTypeList = NULL;
		mmmBlkEntry* pMem = NULL;

		if( mmmVar.mmmOK  ) return FALSE;

		if( nBlockTypes >= MMM_MAX_BLOCK_TYPES ) 
			nBlockTypes = MMM_MAX_BLOCK_TYPES;

		//check memory scheme
		for( i = 0 ; i< nBlockTypes; i ++ )
		{
			if( pFitSchem[i].nBlockSize == 0 || pFitSchem[i].nBlockNum == 0 )
			{
				printf("memory scheme eror: must be non-zeor\r\n");
				return FALSE;
			}
		}

		//first get the real physical memory size
		for( i = 0 ; i< nBlockTypes; i ++ )
		{
			ulNeedMaxPhySize += pFitSchem[i].nBlockNum * pFitSchem[i].nBlockSize;			
			nAllBlockNum += pFitSchem[i].nBlockNum;
		}
		
		ulNeedMaxPhySize += nBlockTypes * sizeof(mmmBlkTypeEntry);  //add list size
		ulNeedMaxPhySize += nAllBlockNum * sizeof(mmmBlkEntry); //add mem list size

		//allocate the memory
		mmmVar.PhyAddr_mmm =   malloc( ulNeedMaxPhySize );
		if( mmmVar.PhyAddr_mmm == NULL ) return FALSE;

		#if MMM_DEBUG
			printf("ulNeedMaxPhySize = %d, nAllBlockNum = %d, base_Addr=%p\r\n", ulNeedMaxPhySize, nAllBlockNum, mmmVar.PhyAddr_mmm );
		#endif

		//begin setup list
		mmmVar.PhyAddr_TypeEntry = mmmVar.PhyAddr_mmm;
		mmmVar.PhyAddr_BlkEntry = mmmVar.PhyAddr_TypeEntry + 2 * nBlockTypes * sizeof(mmmBlkTypeEntry);
	    mmmVar.PhyAddr_Block = mmmVar.PhyAddr_BlkEntry  + nAllBlockNum * sizeof(mmmBlkEntry);

		pListEntryPhyAddr = mmmVar.PhyAddr_TypeEntry;
		pMemEntryPhyAddr = mmmVar.PhyAddr_BlkEntry ;
		pBlockAddr = mmmVar.PhyAddr_Block;

		lstInit( &mmmVar.FreeList);
		lstInit( &mmmVar.UsedList);
		mmmVar.mmmSemID   = semBCreate( SEM_Q_PRIORITY, SEM_EMPTY ); //mutil_mute protection
		if( mmmVar.mmmSemID == NULL )
		{
			printf("Error: semMCreate fail in InitMmmList\r\n");
			free( mmmVar.PhyAddr_mmm );
			return FALSE;
		}
		semGive( mmmVar.mmmSemID    );

		//Free list
		for( i = 0 ; i< nBlockTypes; i ++ )
		{	
			//First add one block type entry
			pFreeBlkTypeList =(mmmBlkTypeEntry*) pListEntryPhyAddr;

			pFreeBlkTypeList->nBlockNum = pFitSchem[i].nBlockNum;
			pFreeBlkTypeList->nBlockSize = pFitSchem[i].nBlockSize;
			pFreeBlkTypeList->PhyAddr_Begin = pBlockAddr;
			pFreeBlkTypeList->PhyAddr_End = pBlockAddr + pFitSchem[i].nBlockNum * pFitSchem[i].nBlockSize-1;
			pFreeBlkTypeList->bfFixed = pFitSchem[i].bfFixed;


			pFreeBlkTypeList->ulRequestTimes = 0;
			pFreeBlkTypeList->ulRespondOKTimes = 0;
			pFreeBlkTypeList->ulRespondBadTimes = 0;
			pFreeBlkTypeList->ulRespondFailTimes = 0;

			//must add it at the end of mmmVar.FreeList
			lstAdd(&mmmVar.FreeList, &(pFreeBlkTypeList->Node) );

			lstInit( &pFreeBlkTypeList->pmemEntryList);

			//Then add all its memory entry						
			for( j= 0; j<(int)pFitSchem[i].nBlockNum; j++ )
			{
				pMem = (mmmBlkEntry *) pMemEntryPhyAddr;

				pMem->BlockAddr = pBlockAddr;
				lstAdd(&pFreeBlkTypeList->pmemEntryList, &(pMem->Node) );
				
				//renew pMemEntryPhyAddr
				pMemEntryPhyAddr += sizeof(mmmBlkEntry);

				//renew pBlockAddr
				pBlockAddr += pFitSchem[i].nBlockSize;
			}

			//renew pListEntryPhyAddr
			pListEntryPhyAddr += sizeof(mmmBlkTypeEntry);

		}

		//used list
		for( i = 0 ; i< nBlockTypes; i ++ )
		{	
			//First add one block type entry
			pUsedBlkTypeList =(mmmBlkTypeEntry*) pListEntryPhyAddr;

			pUsedBlkTypeList->nBlockNum = pFitSchem[i].nBlockNum;
			pUsedBlkTypeList->nBlockSize = pFitSchem[i].nBlockSize;
			
			//must add it at the end of mmmVar.FreeList
			lstAdd(&mmmVar.UsedList, &(pUsedBlkTypeList->Node) );

			lstInit( &pUsedBlkTypeList->pmemEntryList);
			//Note, now there is no member of pUsedBlkTypeList->pmemEntryList

			//renew pListEntryPhyAddr
			pListEntryPhyAddr += sizeof(mmmBlkTypeEntry);
		}

		mmmVar.mmmOK   = TRUE;

		return TRUE;
			
	}


	LOCAL mmmBlkEntry *mmm_GetFreeEntry( size_t nBytes, char *ModuleName )
	{		
		mmmBlkTypeEntry* pFreeBlkTypeList = NULL;
		mmmBlkTypeEntry* pUsedBlkTypeList = NULL;
		mmmBlkTypeEntry* pFitFreeBlkTypeList = NULL;
		mmmBlkEntry* pMem = NULL;
		int nPrevBlockSize = 0;
		BOOL bfJustFitBlockSize = FALSE;
		int ModuleNameLen;
		
		
		if( !mmmVar.mmmOK  ) return NULL;
		
		if( semTake(mmmVar.mmmSemID, WAIT_FOREVER)== ERROR ) return NULL;
		
		pFreeBlkTypeList = (mmmBlkTypeEntry*)lstFirst( &mmmVar.FreeList);
		pUsedBlkTypeList = (mmmBlkTypeEntry*)lstFirst( &mmmVar.UsedList );

		if( pFreeBlkTypeList == NULL || pUsedBlkTypeList == NULL )
		{
			printf("pFreeBlkTypeList or pUsedBlkTypeList is null in mmm_GetFreeEntry\r\n");
			goto EXIT_ERROR;
		}

		nPrevBlockSize = 0;
		while( pFreeBlkTypeList != NULL )
		{
			//printf("mmm_GetFreeEntry: realSize = %d--Reqsize = %d\r\n", pFreeBlkTypeList->nBlockSize, nBytes );
			
			if( (pFreeBlkTypeList->nBlockSize >= nBytes) && (nPrevBlockSize <= nBytes) )
			{
				pFreeBlkTypeList->ulRequestTimes++;

				bfJustFitBlockSize = TRUE;
				pFitFreeBlkTypeList = pFreeBlkTypeList;
			}
			else
				bfJustFitBlockSize = FALSE;

			if( pFreeBlkTypeList->nBlockSize < nBytes ) goto CONTINUE_NEXT;
			if( pFreeBlkTypeList->bfFixed && !bfJustFitBlockSize ) 
			{				
				goto CONTINUE_NEXT;
			}

			pMem = (mmmBlkEntry*) lstFirst( &pFreeBlkTypeList->pmemEntryList);
			if( pMem == NULL)
			{  //this block is empty
				goto CONTINUE_NEXT;
			}

			//get a free block, then set some information 
			#if MMM_RECORD_MODULENAME
				if( ModuleName != NULL ) 
				{	
					ModuleNameLen = (strlen(ModuleName) > MMM_MAX_MODULENAME_LEN-1) ?
						MMM_MAX_MODULENAME_LEN -1: strlen(ModuleName);
					strncpy( pMem->ModuleName, ModuleName, ModuleNameLen );
					pMem->ModuleName[ModuleNameLen] = 0;
				}
				else pMem->ModuleName[0] = 0; 
			#endif	

			#if MMM_RECORD_TIME
				pMem->ulTicks = tickGet();
			#endif		
			
			#if MMM_RECORD_TASK
				pMem->taskID = taskIdSelf();
			#endif

			#if MMM_RECORD_REQUEST_SIZE
				pMem->nReqBlkSize = nBytes;
			#endif	

			//delete it from free-memory-list
			lstDelete( &pFreeBlkTypeList->pmemEntryList, &pMem->Node ); 

			#if 0
				//add it at the end of used-free-memory 
				lstAdd(&pUsedBlkTypeList->pmemEntryList,&(pMem->Node) );
			#else
				//add it at the head of used-free-memory: it will much quich to enquire list
				lstInsert(&pUsedBlkTypeList->pmemEntryList,NULL,&(pMem->Node) );
			#endif


			if( bfJustFitBlockSize ) 
				pFreeBlkTypeList->ulRespondOKTimes++;
			else 
				pFreeBlkTypeList->ulRespondBadTimes++;

			//printf("pMem ok\r\n");

			goto EXIT_OK;


		CONTINUE_NEXT:
			nPrevBlockSize = pFreeBlkTypeList->nBlockSize;
			pFreeBlkTypeList = (mmmBlkTypeEntry*)lstNext( &pFreeBlkTypeList->Node);
			pUsedBlkTypeList = (mmmBlkTypeEntry*)lstNext( &pUsedBlkTypeList->Node); 
		}

		#if MMM_DEBUG
			printf("MMM have no free menory left!\r\n"); 
		#endif

EXIT_ERROR:
		if( pFitFreeBlkTypeList != NULL )
			pFitFreeBlkTypeList->ulRespondFailTimes ++;
		semGive( mmmVar.mmmSemID);
		return NULL;
		
EXIT_OK:     		
		semGive( mmmVar.mmmSemID);
		return pMem; 
	}

	void *mmm_malloc( size_t nBytes, char *ModuleName )
 	{
		mmmBlkEntry* pMem;
				
		#if MMM_DEBUG		
			if( ModuleName != NULL )  
				if( strlen(ModuleName) >= MMM_MAX_MODULENAME_LEN - 1 )
             printf("ModuleName too long(%s). its max length is %d\r\n", ModuleName, MMM_MAX_MODULENAME_LEN - 1 );
		#endif

		if( !mmmVar.mmmOK  ) 
		{
			#if MMM_DEBUG
				printf("mmmOK error in mmm_malloc\r\n");
			#endif
			
			return NULL;
		}
		
		if( ModuleName != NULL )
			pMem = mmm_GetFreeEntry( nBytes, ModuleName);
		else
			pMem = mmm_GetFreeEntry( nBytes, "");

 		if( pMem == NULL ) return NULL;
		
 		return pMem->BlockAddr;
 	}

	/*Note, mmm_realloc only allow small memory block to larget memory block. Otherwise, mmm_realloc
	  will still use the old memory block */
	void *mmm_realloc( void *pBlock, size_t nBytes, char *ModuleName )
	{
		
		mmmBlkEntry* pMem = NULL;
		int nBlockSize = 0;
		
		if( !mmmVar.mmmOK  ) 
		{
			return NULL;
		}

		#if MMM_DEBUG		
			if( ModuleName != NULL )  if( strlen(ModuleName) >= MMM_MAX_MODULENAME_LEN - 1 )
            	printf("ModuleName too long. its max length is %d\r\n", MMM_MAX_MODULENAME_LEN - 1 );
		#endif

		/*first check if old block is old enough*/		
		if( semTake(mmmVar.mmmSemID, WAIT_FOREVER)== ERROR ) 
		{
			mmm_free( pBlock, ModuleName );
			return NULL;
		}		
		nBlockSize = GetBlockSize_private( pBlock);
		if( nBlockSize < 0 ) 
		{
			printf("mmm realloc fail: Maybe some application memory pointer error\r\n");
			semGive(mmmVar.mmmSemID);
			return NULL;			
		}
		
		semGive(mmmVar.mmmSemID);
		
		if((int) nBlockSize >= (int)nBytes )
		{ //old block is big enough			
			//printf("mmm_reaalloc ok: %s--addr %p\r\n",ModuleName, pBlock);
			
			return pBlock;
		}

		//printf("need to renew a block\r\n");

		//old memory block is not big enough, so allocate a new big memory block
		pMem = mmm_GetFreeEntry( nBytes, ModuleName);
		if( pMem == NULL ) 
		{
			mmm_free( pBlock, ModuleName );
			return NULL;
		}

		//copy old content to new memory 
		if( semTake(mmmVar.mmmSemID, WAIT_FOREVER)== ERROR ) 
		{
			mmm_free( pBlock, ModuleName );
			mmm_free((void *) pMem->BlockAddr, ModuleName );
			return NULL;			
		}

		memcpy( pMem->BlockAddr, pBlock, nBlockSize );
		
		semGive( mmmVar.mmmSemID );

		//printf("mmm_realloc ok:%s-- addr %p, size %d\r\n", ModuleName, pMem->BlockAddr, nBytes );

		return pMem->BlockAddr;

	}

	void mmm_free( void *pBlock, char *ModuleName)
	{	
		//#define MMM_NEED_SORT
		int nlistNo = 0;
		mmmBlkEntry* pMem = NULL;	

		#ifdef MMM_NEED_SORT
			mmmBlkEntry* pTmpMem = NULL;		
			mmmBlkEntry* pTmpPrevMem = NULL;	
		#endif
		
		mmmBlkTypeEntry* pFreeBlkTypeList = NULL;
		mmmBlkTypeEntry* pUsedBlkTypeList = NULL;
		
		if( pBlock == NULL ) 
		{
			//printf("want to free NULL\r\n");
			return;
		}
		
		if( !mmmVar.mmmOK  ) 
		{
			//printf("want to mmm_free, but mmm status not ok\r\n");
			return ;
		}

		if( semTake(mmmVar.mmmSemID, WAIT_FOREVER)== ERROR ) 
		{
			printf("semtake error in mmm_free\r\n");
			return;
		}

		//for quick seach, here we use some magic method according to physical memory address
		nlistNo = GetListNo_private(pBlock);		
		if( nlistNo == - 1) 
		{
			printf("mmm free fail: maybe some application memory ponter error(%p)\r\n", pBlock );
			goto EXIT;
		}
		//printf("mmm_free: nlistNo = %d\r\n", nlistNo );
		
		pFreeBlkTypeList = (mmmBlkTypeEntry*)lstNth( &mmmVar.FreeList, nlistNo );
		pUsedBlkTypeList = (mmmBlkTypeEntry*)lstNth( &mmmVar.UsedList, nlistNo );
		if( pFreeBlkTypeList == NULL || pUsedBlkTypeList == NULL) 
		{			
			printf("mmm_free error1\r\n");
			goto EXIT;
		}
		//printf("pBlock = %p PhyAddr_Begin %p End %p\r\n", pBlock, pFreeBlkTypeList->PhyAddr_Begin, pFreeBlkTypeList->PhyAddr_End );

		pMem = (mmmBlkEntry*)lstFirst( &pUsedBlkTypeList->pmemEntryList);
		while( pMem != NULL )
		{
			if( pBlock == pMem->BlockAddr )
			{	
			#if MMM_RECORD_MODULENAME
				if( strncmp(ModuleName, pMem->ModuleName, MMM_MAX_MODULENAME_LEN - 1) == 0 ) 
			#endif			
				{					
					//delete it from used-memory-list
					lstDelete( &pUsedBlkTypeList->pmemEntryList, &pMem->Node ); 

					
					#ifndef MMM_NEED_SORT  //add it at the end of free-memory-list
						lstAdd(&pFreeBlkTypeList->pmemEntryList, &(pMem->Node) );
					#else //put in order according to physical memory address: First small, then bigger, and biggest 
						pTmpMem = (mmmBlkEntry*)lstFirst( &pFreeBlkTypeList->pmemEntryList);
						if( pTmpMem == NULL ) 
						{  //only one entry, add first
							lstAdd(&pFreeBlkTypeList->pmemEntryList, &(pMem->Node) );
							goto EXIT;
						}
						while( pTmpMem != NULL )
						{
							if( pTmpMem->BlockAddr < pMem->BlockAddr )
							{
								pTmpPrevMem = pTmpMem;
								pTmpMem = (mmmBlkEntry*)lstNext( &pTmpMem->Node);
								continue;
							}
							
							break;
						}
						
						//add after pTmpPrevMem
						if( pTmpPrevMem == NULL )
							lstAdd(&pFreeBlkTypeList->pmemEntryList, &(pMem->Node) );
						else
							lstInsert(&pFreeBlkTypeList->pmemEntryList, &pTmpPrevMem->Node, &(pMem->Node) );

					#endif

					goto EXIT;
				}
			#if MMM_RECORD_MODULENAME
				else
				{
					printf("taskname not much: record name %s-mmm_free name %s\r\n", ModuleName, pMem->ModuleName );
				}
			#endif			
			
			}
			
			pMem  = (mmmBlkEntry*)lstNext( &pMem->Node);
		}		

	EXIT:
		
		semGive(mmmVar.mmmSemID);		
	}



	/*show used memory according to its modulenamt */
	void mmm_ShowViaModuleName(char *ModuleName, int nType)
	{
		#if MMM_RECORD_MODULENAME		
			int num = 0;
			mmmBlkTypeEntry* pUsedBlkTypeList = NULL;
			mmmBlkEntry* pMemEntry = NULL;
		#endif
			ULONG ulCurrentTicks = tickGet();

		if( !mmmVar.mmmOK  ) return ;

		#if MMM_RECORD_MODULENAME
		
			if( semTake(mmmVar.mmmSemID, WAIT_FOREVER)== ERROR ) return;

			if( nType == 1 )
			{
				pUsedBlkTypeList = (mmmBlkTypeEntry*)lstFirst( &mmmVar.UsedList);
				while(pUsedBlkTypeList != NULL )
				{
					pMemEntry = (mmmBlkEntry*) lstFirst( &pUsedBlkTypeList->pmemEntryList);
					
					while( pMemEntry != NULL )
					{		
						if( strncmp( ModuleName, pMemEntry->ModuleName, MMM_MAX_MODULENAME_LEN - 1 )== 0 )
						{
							num ++;	
						}				
						pMemEntry = (mmmBlkEntry*)lstNext( &pMemEntry->Node);
					}

					pUsedBlkTypeList = (mmmBlkTypeEntry*)lstNext( &pUsedBlkTypeList->Node);
				}			
			}
			else if( nType == 2 )
			{
				pUsedBlkTypeList = (mmmBlkTypeEntry*)lstFirst( &mmmVar.UsedList);

				PrintHeader();
				
				while(pUsedBlkTypeList != NULL )
				{
					pMemEntry = (mmmBlkEntry*) lstFirst( &pUsedBlkTypeList->pmemEntryList);

					while(pMemEntry != NULL )
					{
						if( strncmp( ModuleName, pMemEntry->ModuleName, MMM_MAX_MODULENAME_LEN - 1)== 0 )
						{
							printf(" %-4d(%p) ", num , pMemEntry->BlockAddr );
							num ++;
						
							#if MMM_RECORD_MODULENAME	
								printf(" %8s", pMemEntry->ModuleName);
							#endif

							#if MMM_RECORD_TIME
								printf(" %9ld", (ulCurrentTicks - pMemEntry->ulTicks)/mmmVar.sysTickRate);
							#endif
							
							#if MMM_RECORD_TASK
								printf(" %9d", pMemEntry->taskID);
							#endif

							#if MMM_RECORD_REQUEST_SIZE
								printf(" %7d", pMemEntry->nReqBlkSize);								
							#endif	

							printf("\r\n");
						}
						
						//next blk
						pMemEntry = (mmmBlkEntry*)lstNext( &pMemEntry->Node);
					}				

					pUsedBlkTypeList = (mmmBlkTypeEntry*)lstNext( &pUsedBlkTypeList->Node);
				}

				PrintHeader();
				
			}

			semGive(mmmVar.mmmSemID);

			printf("Module %s have memory block %d\r\n", ModuleName,num );
		#endif	
	
	}

	/* 1--show simple used memory blocks; 2---show detail used memory blocks*/	
	void mmm_ShowUsedMem(int nType)
	{
		int i, j;
		mmmBlkTypeEntry* pUsedBlkTypeList = NULL;
		mmmBlkEntry* pMemEntry = NULL;
		ULONG ulCurrentTicks = tickGet();

		if( !mmmVar.mmmOK  ) return ;
		
		if( semTake(mmmVar.mmmSemID, WAIT_FOREVER)== ERROR ) return;

		pUsedBlkTypeList = (mmmBlkTypeEntry*)lstFirst( &mmmVar.UsedList);

		if( nType == 1 )
		{
			i = 0; 
			
			while(pUsedBlkTypeList != NULL )
			{
				printf("\r\nUsed List %2d: Blk size %6d---Blk num %4d\r\n",  i , pUsedBlkTypeList->nBlockSize, lstCount( &pUsedBlkTypeList->pmemEntryList) );

				pUsedBlkTypeList = (mmmBlkTypeEntry*)lstNext( &pUsedBlkTypeList->Node);
				i++;
			}
			
		}		
		else if( nType == 2 )
		{
			i = 0; j = 0;

			PrintHeader();
			
			while(pUsedBlkTypeList != NULL )
			{
				if( lstCount( &pUsedBlkTypeList->pmemEntryList) > 0 )
				{
					pMemEntry = (mmmBlkEntry*) lstFirst( &pUsedBlkTypeList->pmemEntryList);					
					
					while(pMemEntry != NULL )
					{
						printf("  %-4d(%p) ", j , pMemEntry->BlockAddr );
						
						#if MMM_RECORD_MODULENAME
							printf(" %-8s", pMemEntry->ModuleName);
						#endif

						#if MMM_RECORD_TIME
							printf(" %9ld", (ulCurrentTicks - pMemEntry->ulTicks)/mmmVar.sysTickRate);
						#endif
						
						#if MMM_RECORD_TASK
							printf(" %9d", pMemEntry->taskID);
						#endif

						#if MMM_RECORD_REQUEST_SIZE
							printf("  %7d", pMemEntry->nReqBlkSize);						
						#endif	

						printf("\r\n");
						
						//next blk
						pMemEntry = (mmmBlkEntry*)lstNext( &pMemEntry->Node);
						j ++;
					}
				}

				pUsedBlkTypeList = (mmmBlkTypeEntry*)lstNext( &pUsedBlkTypeList->Node);
				i++;
			}

			PrintHeader();
		}


		semGive(mmmVar.mmmSemID  );

	}


	/* 1--show simple free memory blocks; 2---show detail free memory blocks*/	
	void mmm_ShowFreeMem(int nType)
	{
		int i, j;
		mmmBlkTypeEntry* pFreeBlkTypeList = NULL;
		mmmBlkEntry* pMemEntry = NULL;


		if( !mmmVar.mmmOK  ) return;
		
		if( semTake(mmmVar.mmmSemID, WAIT_FOREVER)== ERROR ) return;

		pFreeBlkTypeList = (mmmBlkTypeEntry*)lstFirst( &mmmVar.FreeList);
		
		if( nType == 1 )
		{
			i = 0;

			printf("Free block summary: \r\n");
			while(pFreeBlkTypeList != NULL )
			{
				printf("%2d: %s %d, blocks %d--left %d, Reqest: %ld, Fit resp: %ld, no fit: %ld, fail %ld\r\n", 
					i ,pFreeBlkTypeList->bfFixed? "SIZE":"size", pFreeBlkTypeList->nBlockSize, pFreeBlkTypeList->nBlockNum, 
					lstCount( &pFreeBlkTypeList->pmemEntryList), pFreeBlkTypeList->ulRequestTimes, 
					pFreeBlkTypeList->ulRespondOKTimes, pFreeBlkTypeList->ulRespondBadTimes, pFreeBlkTypeList->ulRespondFailTimes );

				pFreeBlkTypeList = (mmmBlkTypeEntry*)lstNext( &pFreeBlkTypeList->Node);
				i++;
			}
		}
		else if( nType == 2 ) 			
		{
			i = 0; j = 0;
			while(pFreeBlkTypeList != NULL )
			{
				printf("Free List %2d: size %6d num %4d (of %4d) from %p to %p\r\n", 
					i , pFreeBlkTypeList->nBlockSize, lstCount( &pFreeBlkTypeList->pmemEntryList), pFreeBlkTypeList->nBlockNum, pFreeBlkTypeList->PhyAddr_Begin, pFreeBlkTypeList->PhyAddr_End);

				if( lstCount(&pFreeBlkTypeList->pmemEntryList) > 0 )
				{
					pMemEntry = (mmmBlkEntry*) lstFirst( &pFreeBlkTypeList->pmemEntryList);
					while(pMemEntry != NULL )
					{
						printf("  %-4d: address %p \r\n", j , pMemEntry->BlockAddr );
						
						pMemEntry = (mmmBlkEntry*)lstNext( &pMemEntry->Node);
						j++;
					}
				}
				printf("\r\n");

				pFreeBlkTypeList = (mmmBlkTypeEntry*)lstNext( &pFreeBlkTypeList->Node);
				i++;
			}
		}

		semGive(mmmVar.mmmSemID  );
	}


    /********************************************************************************
     Note, First List no is 1, and so on with step 1
          For quick search, here we use really memory address to search the List
          otherwise -1 if error
    ********************************************************************************/
    LOCAL int GetListNo_private(void *pBlock)
	{
		mmmBlkTypeEntry *pFreeBlkTypeList = (mmmBlkTypeEntry*)lstFirst( &mmmVar.FreeList);
		int  i = 1;
		
		if( pFreeBlkTypeList == NULL ) 
		{
			printf("GetListNo_private error\r\n");
			return -1;
		}
		while( pFreeBlkTypeList != NULL )
		{
			//printf("pBlock = %p PhyAddr_Begin %p End %p\r\n", pBlock, pFreeBlkTypeList->PhyAddr_Begin, pFreeBlkTypeList->PhyAddr_End );
			
			if((ULONG) pBlock >= (ULONG)pFreeBlkTypeList->PhyAddr_Begin && 
				(ULONG)pBlock <=  (ULONG)pFreeBlkTypeList->PhyAddr_End)
			{
				return i;
			}

			pFreeBlkTypeList = (mmmBlkTypeEntry*)lstNext( &pFreeBlkTypeList->Node);
			i++;
		}
		
		return -1;
	}

    /********************************************************************************
     Note, For quick search, here we use really memory address to search the block's real size
    ********************************************************************************/
    LOCAL int GetBlockSize_private(void *pBlock)
	{
		mmmBlkTypeEntry *pFreeBlkTypeList = (mmmBlkTypeEntry*)lstFirst( &mmmVar.FreeList);
		int  i = 1;
		
		if( pFreeBlkTypeList == NULL ) return -1;
		while( pFreeBlkTypeList != NULL )
		{
			if( (ULONG)pBlock>= (ULONG)pFreeBlkTypeList->PhyAddr_Begin && 
				(ULONG)pBlock<= (ULONG)pFreeBlkTypeList->PhyAddr_End)
			{
				return pFreeBlkTypeList->nBlockSize;
			}

			pFreeBlkTypeList = (mmmBlkTypeEntry*)lstNext( &pFreeBlkTypeList->Node);
			i++;
		}
		
		return -1;
	}

    
    void PrintHeader()
	{
		printf("  No   Address  "); 
		
		#if MMM_RECORD_MODULENAME 
			printf(" ModuleName ");
		#endif
		
		#if MMM_RECORD_TIME
			printf("Time(sec) ");
		#endif

		#if MMM_RECORD_TASK
			printf("  taskID    ");
		#endif
		
		#if MMM_RECORD_REQUEST_SIZE
			printf(" RequestSize");
		#endif
		
		printf("\r\n");
	}
	


    LOCAL int mmmComparQSort( mmm_fit_schem *First, mmm_fit_schem *Second)
	{
		if( First->nBlockSize > Second->nBlockSize ) return 1;
		else return -1;
	}

#else	
	BOOL mmm_init(MMM_CFG *pmmmCfg)
	{
		return TRUE;
	}

	void *mmm_malloc( size_t nBytes, char *ModuleName )
	{
		return malloc( nBytes );
	}	

	void *mmm_realloc( void *pBlock, size_t nBytes, char *ModuleName )
	{
		return realloc(pBlock, nBytes);
	}

	void mmm_free( void *pBlock, char *ModuleName)
	{
		free(pBlock);
	}

	void mmm_ShowUsedMem(int nType)
	{
	}
	void mmm_ShowFreeMem(int nType)
	{
	}
	
#endif


