/***************************************************************************
 *   MODULE NAME    : brdg_filter.c                                        *
 ***************************************************************************
 *   Copyright 2001 Conexant Systems, Inc. as unpublished work             *
 *   All Rights Reserved                                                   *
 *                                                                         *
 *   The information contained herein is the confidential property         *
 *   of Conexant Systems, Inc..  The use, copying, transfer or             *
 *   disclosure of such information is prohibited except by express        *
 *   written agreement with Conexant Systems, Inc..                        *
 *                                                                         *
 *   Original Written By  :    C W Chen                                    *
 *   Original Date        :    10/10/2001                                  *
 ***************************************************************************/
/***************************************************************************
 * This file implements all the APIs for accessing filtering rule table for*
 * Brdg Filtering function in Hasbani project. We provide this layer for   *
 * multiple accessing this table.										   *
 * All the data structure defintions are in header file "brdg_filter.h"    *
 ***************************************************************************/

#include <Vxworks.h>
#include <taskLib.h>
#include "brdg_filter.h"
#include "protoutils.h"
#include "JxRegSrv.h"
#include "JxCfgMgr.h"

#ifdef SYSLOG_ENABLED
#include "logdrv.h"
#endif


#define 	SOFTWARE_FILTER

/***************************************************************************/
/*                          GLOBAL VARIABLE DECLARATION                    */
/***************************************************************************/
LOCAL STRUCT_BrdgFilterRuleTbl     FILTER_rule_table;

LOCAL int curr_group, curr_rule;

int brdg_filter_is_on;      /* put this global variable for faster accessing*/
							/* the mac filter function is on or not			*/
							/* Instead of accessing through single entry	*/
							/* API, we provide this global variable 		*/
							/* synchronized with the mac filter enable or   */
							/* disable. Other tasks can just read this 		*/
							/* variable to know the mac filter function is  */
							/* on or off. This variable will ONLY be 		*/
                            /* changed by  brdgFILTER_tbl_init(),           */
                            /* brdgFILTER_enable() and brdgFILTER_disable() */


/***************************************************************************/
/*                          LOCAL FUNCTION DECLARATION                     */
/***************************************************************************/
/*------------------------------------------------------------*/
/*               functions in ruleTblAccessFunc[]             */
/*------------------------------------------------------------*/
LOCAL FILTER_STATUS     brdgFILTER_tbl_init(void* dummy);
LOCAL FILTER_STATUS     brdgFILTER_enable(void* dummy);
LOCAL FILTER_STATUS     brdgFILTER_disable(void* dummy);
LOCAL FILTER_STATUS     brdgFILTER_tbl_addEntry(void* entry);
LOCAL FILTER_STATUS     brdgFILTER_tbl_delEntry(void* dummy);
LOCAL FILTER_STATUS     brdgFILTER_tbl_modEntry(void* entry);
LOCAL FILTER_STATUS     brdgFILTER_tbl_dataCheck(void* entry);
LOCAL FILTER_STATUS     brdgFILTER_tbl_getEntry(void* entry);
LOCAL FILTER_STATUS     brdgFILTER_tbl_getTotal(void* total);
LOCAL FILTER_STATUS     brdgFILTER_tbl_getStatus(void* status);


/*-------------------------------------------------------------*/
/*		internal routines for direct accessing address table   */
/*															   */
/* These routines SHOULD be called by functions in             */
/* ruleTblAccessFunc[] ONLY!!!                                 */
/*-------------------------------------------------------------*/
LOCAL int 		_rule_compare( UINT8* target_data, UINT8* key_rule,
								UINT8 length, UINT8* mask);
LOCAL int		_binary_recursive( UINT8* target, int bottom,
											int top, int *near_index);
LOCAL int		_bSearch( UINT8* target, int bottom, int top);
LOCAL STATUS    _ruleTbl_checkData( UINT8* data, int* entry_loc);
LOCAL STATUS    _ruleTbl_deleteEntry( UINT8 entry_index);
LOCAL STATUS    _ruleTbl_getEntry(STRUCT_WebBrdgFilterTbl* buf);

LOCAL int parseStr2Hex(char *str, UINT8* data, int hexcnt);
LOCAL void formatHex2Str(char* buf, char* str, int len);
LOCAL char xdigit2i(char digit);
LOCAL char i2xdigit(char digit);


/***************************************************************************/
/*                          LOCAL VARIABLE DECLARATION                     */
/***************************************************************************/
LOCAL SEM_ID		filterTbl_access_sem=NULL;


/*--------------------------------------------------------*/
/* Table of Function Pointers for Accessing Address Table */
/*														  */
/* The order of these function pointers SHOULD be the same*/
/* as "func_num" defined in "mac_table.h"...              */
/*--------------------------------------------------------*/
LOCAL FILTER_STATUS ( *brdgFilterAccessFunc[] )(	void* param )=
{

#ifdef SOFTWARE_FILTER

    brdgFILTER_tbl_init,
    brdgFILTER_enable,
    brdgFILTER_disable,
    brdgFILTER_tbl_addEntry,
    brdgFILTER_tbl_delEntry,
    brdgFILTER_tbl_dataCheck,
    brdgFILTER_tbl_getEntry,
    brdgFILTER_tbl_getTotal,
    brdgFILTER_tbl_getStatus,
#else

#endif




	NULL
};




/****************************************************************************
FUNCTION NAME   : FILTER_STATUS     brdgFILTER_CTRL(int func_num, int grpn,
                                                    int rn,  void* param)

DESCRIPTION     : This routine provides all methods to access filter address table.
				  To provide multiple entries for different tasks, it
				  serializes all the different requests invoking this routine.

INPUT           : int func_num: function number, this is an index indicating
                                which behavior the invoking routine will do.
                                All the function number definition are
                                defined in "mac_table.h".
                  int   grpn:   group# of brdg filter tabl
                  int   rn:     group# of brdg filter tabl
                  void* param:  passed parameter. it will have different
                                data structure for each indivisual function.

OUTPUT          : None

RETURN          : FILTER_STATUS
****************************************************************************/
FILTER_STATUS   brdgFILTER_CTRL(int func_num, int grpn, int rn, void* param)
{
	FILTER_STATUS 	rtn_status;

    /* create semaphore */
	if( filterTbl_access_sem==NULL )
	{
		filterTbl_access_sem=semBCreate(SEM_Q_PRIORITY, SEM_FULL);

		/* semaphore creation fail */
		if( filterTbl_access_sem==NULL )
			return FILTER_SEM_ERROR;
	}

	/* take semaphore */
	if( ERROR==semTake(filterTbl_access_sem, 500) )
		return FILTER_SEM_ERROR;

	/* parse function pointer according to passed func_num */
    if( brdgFilterAccessFunc[func_num]!=NULL )
	{
        curr_group = grpn;      /* Current group# */
        curr_rule = rn;         /* Current rule# */
        rtn_status=(brdgFilterAccessFunc[func_num])(param);
        semGive(filterTbl_access_sem);


		/* return status is directly from internal rioutine */
		return rtn_status;
	}

    semGive(filterTbl_access_sem);
	return FILTER_FUNC_ERROR;

} /* end of brdgFILTER_CTRL() */



/****************************************************************************
FUNCTION NAME   : LOCAL FILTER_STATUS   brdgFILTER_tbl_init(void* dummy)

DESCRIPTION     : This routine will be invoked only by brdgFILTER_CTRL().
				  It will initialize the mac address table structure. It
                  should be called ( through brdgFILTER_CTRL() ) during
				  system startup.

INPUT           : void* dummy: Not used by this routine.

OUTPUT          : None

RETURN          : FILTER_OK - always return OK...
****************************************************************************/
FILTER_STATUS brdgFILTER_tbl_init(void* dummy)
{

    int i;
    /* all we need to set are "max_tbl_size" and "currt_tbl_size"... */
	/* no need to fill in any default values for each fields in the  */
	/* table... these two indexes will handle the table precisely... */
    FILTER_rule_table.filter_function=BRDG_FILTER_OFF;
    FILTER_rule_table.max_tbl_size=MAX_FILTER_RULE_TBL_SIZE;
    FILTER_rule_table.currt_tbl_size=0;

	/* save this filter function status to global variable for  */
	/* faster reference in snarf...								*/
    brdg_filter_is_on=FILTER_rule_table.filter_function;

	return FILTER_OK;

} /* end of brdgFILTER_tbl_init() */




/****************************************************************************
FUNCTION NAME   : LOCAL FILTER_STATUS   brdgFILTER_enable(void* dummy)

DESCRIPTION     : This routine will be invoked only by brdgFILTER_CTRL().
				  It will enable mac filtering function.

INPUT           : void*	dummy: Not used by this routine.

OUTPUT          : None

RETURN          : FILTER_OK
****************************************************************************/
LOCAL FILTER_STATUS brdgFILTER_enable(void* dummy)
{
    FILTER_rule_table.filter_function=BRDG_FILTER_ON;

	/* save this filter function status to global variable for  */
	/* faster reference in snarf...								*/
    brdg_filter_is_on=FILTER_rule_table.filter_function;

	return FILTER_OK;

} /* end of brdgFILTER_enable() */


/****************************************************************************
FUNCTION NAME   : LOCAL FILTER_STATUS   brdgFILTER_disable(void* dummy)

DESCRIPTION     : This routine will be invoked only by brdgFILTER_CTRL().
				  It will disable mac filtering function.

INPUT           : void* dummy: Not used by this routine.

OUTPUT          : None

RETURN          : FILTER_OK
****************************************************************************/
LOCAL FILTER_STATUS brdgFILTER_disable(void* dummy)
{
    FILTER_rule_table.filter_function=BRDG_FILTER_OFF;

	/* save this filter function status to global variable for  */
	/* faster reference in snarf...								*/
    brdg_filter_is_on=FILTER_rule_table.filter_function;

	return FILTER_OK;

} /* end of brdgFILTER_disable() */



/****************************************************************************
FUNCTION NAME   : LOCAL FILTER_STATUS   brdgFILTER_tbl_getStatus(void* status)

DESCRIPTION     : This routine will be invoked only by brdgFILTER_CTRL().
                  It will get status of filtering function.

INPUT           : void* status: filter function status.

OUTPUT          : None

RETURN          : FILTER_OK
****************************************************************************/
LOCAL FILTER_STATUS brdgFILTER_tbl_getStatus(void* status)
{
    *(int*)status=FILTER_rule_table.filter_function;
    
	/*Synchronize brdg_filter_is_on(used by protocol.c) with filter_function */
	// brdg_filter_is_on=FILTER_rule_table.filter_function;

	return FILTER_OK;

} /* end of brdgFILTER_tbl_getStatus() */



/****************************************************************************
FUNCTION NAME   : LOCAL FILTER_STATUS   brdgFILTER_tbl_addEntry(void* entry)

DESCRIPTION     : This routine will be invoked only by brdgFILTER_CTRL.
                  It add a new entry to the filtering table...

INPUT           : UINT8*  entry: pointer of the mac of corresponding entry.

OUTPUT          : None

RETURN          : FILTER_STATUS
****************************************************************************/
LOCAL FILTER_STATUS brdgFILTER_tbl_addEntry(void* entry)
{

    int len, coln, te;
    STRUCT_WebBrdgFilterTbl* strBuf = (STRUCT_WebBrdgFilterTbl*) entry;

    //int index = FILTER_rule_table.currt_tbl_size;
    int index = (curr_group - 1) * GROUP_SIZE + (curr_rule - 1);

    /* if over group# limit */
    if (curr_rule > GROUP_SIZE)
		return FILTER_ERROR;

	/* if the table full */
    if( curr_group > (MAX_FILTER_RULE_TBL_SIZE/GROUP_SIZE) )
        return FILTER_ERROR;

    if (curr_group > FILTER_rule_table.currt_tbl_size + 1)
		return FILTER_ERROR;

    if (curr_group == FILTER_rule_table.currt_tbl_size + 1)
        FILTER_rule_table.currt_tbl_size = curr_group;

    // parse the entry

    FILTER_rule_table.filterRuleTblEntry[index].filterOffset = strBuf->offset;

    len = strlen(strBuf->value)/2;
    if ((strlen(strBuf->value)) & 1)
        len += 1;

    FILTER_rule_table.filterRuleTblEntry[index].filterLength = len;
    if (len)
    {
        parseStr2Hex(strBuf->mask, FILTER_rule_table.filterRuleTblEntry[index].filterMask, len);
        parseStr2Hex(strBuf->value, FILTER_rule_table.filterRuleTblEntry[index].filterData, len);
        FILTER_rule_table.filterRuleTblEntry[index].filterAction = strBuf->action;
    }

    index = (curr_group-1) * GROUP_SIZE;
    for (coln = 0; coln < GROUP_SIZE; coln++)
    {
        if (FILTER_rule_table.filterRuleTblEntry[index+coln].filterLength)
        {
            for (te = coln+1; te < GROUP_SIZE; te++)
            {
                if (FILTER_rule_table.filterRuleTblEntry[index+te].filterLength)
                    FILTER_rule_table.filterRuleTblEntry[index+coln].filterAction = FILTER_NEXT;
            }
        }
    }
    return FILTER_OK;

} /* end of brdgFILTER_tbl_addEntry() */



/****************************************************************************
FUNCTION NAME   : LOCAL FILTER_STATUS brdgFILTER_tbl_delEntry(int index)

DESCRIPTION     : This routine will be invoked only by brdgFILTER_CTRL.
                  It delete a passed group from the rule table...

INPUT           : void*  data: pointer of the data of corresponding filter
                               rule entry.

OUTPUT          : None

RETURN          : FILTER_STATUS
****************************************************************************/
LOCAL FILTER_STATUS     brdgFILTER_tbl_delEntry(void* index)
{

    if( FILTER_rule_table.currt_tbl_size==0 )
        return FILTER_ERROR;

    if( OK==_ruleTbl_deleteEntry(curr_group) )
        return FILTER_OK;
    else
        return FILTER_ERROR;

} /* end of brdgFILTER_tbl_delEntry() */



/****************************************************************************
FUNCTION NAME   : LOCAL FILTER_STATUS   brdgFILTER_tbl_modEntry(void* entry)

DESCRIPTION     : This routine will be invoked only by brdgFILTER_CTRL.
                  It modify a existing entry in the table...

INPUT           : UINT8*  entry: pointer of the mac of corresponding entry.

OUTPUT          : None

RETURN          : FILTER_STATUS
****************************************************************************/
LOCAL FILTER_STATUS brdgFILTER_tbl_modEntry(void* entry)
{

    int len;
    STRUCT_WebBrdgFilterTbl* strBuf = (STRUCT_WebBrdgFilterTbl*) entry;

    int index = (curr_group - 1) * GROUP_SIZE + (curr_rule - 1);

    // parse the entry

    FILTER_rule_table.filterRuleTblEntry[index].filterOffset = strBuf->offset;

    len = strlen(strBuf->value)/2;
    if ((strlen(strBuf->value)) & 1)
        len += 1;

    FILTER_rule_table.filterRuleTblEntry[index].filterLength = len;
    parseStr2Hex(strBuf->mask, FILTER_rule_table.filterRuleTblEntry[index].filterMask, len);
    parseStr2Hex(strBuf->value, FILTER_rule_table.filterRuleTblEntry[index].filterData, len);

    FILTER_rule_table.filterRuleTblEntry[index].filterAction = strBuf->action;

    return FILTER_OK;

} /* end of brdgFILTER_tbl_modEntry() */


/****************************************************************************
FUNCTION NAME   : LOCAL FILTER_STATUS   brdgFILTER_tbl_dataCheck(void* data)

DESCRIPTION     : This routine will be invoked only by brdgFILTER_CTRL.
				  It compare the passed mac address against mac address table.

INPUT           : void*  mac: pointer of the data to compare against rule table.

OUTPUT          : None

RETURN          : FILTER_STATUS
****************************************************************************/
LOCAL FILTER_STATUS brdgFILTER_tbl_dataCheck(void* data)
{
	int entry_loc;


    if( OK==_ruleTbl_checkData((UINT8*)data, &entry_loc) )
        return FILTER_FORWARD_DATA;
	else
        return FILTER_BLOCK_DATA;

} /* end of brdgFILTER_tbl_dataCheck() */



/****************************************************************************
FUNCTION NAME   : LOCAL FILTER_STATUS   brdgFILTER_tbl_getTotal(void* total)

DESCRIPTION     : This routine will be invoked only by brdgFILTER_CTRL().
                  It will get total groups of current filtering table.

INPUT           : void* total: total groups of filter table.

OUTPUT          : None

RETURN          : FILTER_OK
****************************************************************************/
LOCAL FILTER_STATUS brdgFILTER_tbl_getTotal(void* total)
{
    if ( FILTER_rule_table.currt_tbl_size == 0)
            return FILTER_TBL_EMPTY;
    else
        *(int*) total=FILTER_rule_table.currt_tbl_size;

	return FILTER_OK;

} /* end of brdgFILTER_tbl_getTotal() */



/****************************************************************************
FUNCTION NAME   : LOCAL FILTER_STATUS   brdgFILTER_tbl_getEntry(void* tbl)

DESCRIPTION     : This routine will be invoked only by brdgFILTER_CTRL().
                  It will pass one filter entry to the passed structure.

INPUT           : void* entry: the filter entry will be passed in this structure.

OUTPUT          : None

RETURN          : FILTER_OK
****************************************************************************/
LOCAL FILTER_STATUS     brdgFILTER_tbl_getEntry(void* buf)
{
    int len, i;

    int index = (curr_group - 1) * GROUP_SIZE + (curr_rule - 1);

    /* if over group# limit */
    if (curr_rule > GROUP_SIZE)
		return FILTER_ERROR;

    if (OK==_ruleTbl_getEntry((STRUCT_WebBrdgFilterTbl*) buf))
       return FILTER_OK;
   else    return FILTER_ERROR;

} /* end of brdgFILTER_tbl_getEntry() */


LOCAL STATUS    _ruleTbl_getEntry(STRUCT_WebBrdgFilterTbl* buf)
{
    int len;

    int index = (curr_group - 1) * GROUP_SIZE + (curr_rule - 1);

    /* format data */
    buf->offset = FILTER_rule_table.filterRuleTblEntry[index].filterOffset;

    if ((len = FILTER_rule_table.filterRuleTblEntry[index].filterLength) == 0)
        return ERROR;

    formatHex2Str(buf->mask, FILTER_rule_table.filterRuleTblEntry[index].filterMask, len);
    formatHex2Str(buf->value, FILTER_rule_table.filterRuleTblEntry[index].filterData, len);

    buf->action = FILTER_rule_table.filterRuleTblEntry[index].filterAction;

    return OK;

} /* end of brdgFILTER_tbl_getEntry() */



/****************************************************************************
FUNCTION NAME   : LOCAL STATUS _ruleTbl_deleteEntry( UINT8 entry_index)

DESCRIPTION     : This routine will be invoked only by functions in
                  bedgFilterAccessFunc[]. Internal routine for accessing address
				  table.
				  It delete the entry in the table based on passed entry_index.

INPUT           : UINT8 entry_index: index of the entry in address table to delete.

OUTPUT          : None

RETURN          : OK	- suceeded.
				  ERROR - failed.
****************************************************************************/
LOCAL STATUS _ruleTbl_deleteEntry( UINT8 entry_index)
{

    int index, r_count, len, lc;

    /* if table empty, nothing to delete */
    if( FILTER_rule_table.currt_tbl_size==0 )
		return ERROR;

	/* error handling */
    if( entry_index>(FILTER_rule_table.currt_tbl_size) )
		return ERROR;

	/* adjust currt_tbl_size */
    FILTER_rule_table.currt_tbl_size--;

	/* if only one entry, delete it means just adjust currt_tbl_size */
    if( FILTER_rule_table.currt_tbl_size==0 )
		return OK;

    index = (entry_index - 1) * GROUP_SIZE;      /* Start entry of deleted group */

    for (r_count = index; r_count < FILTER_rule_table.currt_tbl_size*GROUP_SIZE; r_count++)
    {
        FILTER_rule_table.filterRuleTblEntry[r_count].filterOffset =
                FILTER_rule_table.filterRuleTblEntry[r_count+GROUP_SIZE].filterOffset;

        FILTER_rule_table.filterRuleTblEntry[r_count].filterLength =
                                                               len =
            FILTER_rule_table.filterRuleTblEntry[r_count+GROUP_SIZE].filterLength;

        for (lc = 0; lc < len ; lc++)
        {
            FILTER_rule_table.filterRuleTblEntry[r_count].filterMask[lc] =
                FILTER_rule_table.filterRuleTblEntry[r_count+GROUP_SIZE].filterMask[lc];
            FILTER_rule_table.filterRuleTblEntry[r_count].filterData[lc] =
                FILTER_rule_table.filterRuleTblEntry[r_count+GROUP_SIZE].filterData[lc];
        }

        FILTER_rule_table.filterRuleTblEntry[r_count].filterAction =
            FILTER_rule_table.filterRuleTblEntry[r_count+GROUP_SIZE].filterAction;
    }

}  /* end of _ruleTbl_deleteEntry() */



/****************************************************************************
FUNCTION NAME   : LOCAL STATUS _ruleTbl_checkData( UINT8* data, int* entry_loc)

DESCRIPTION     : This routine will be invoked only by functions in
                  brdgFilterAccessFunc[]. Internal routine for accessing address
				  table.
                  It checks if the passed data is in the table or not. If yes,
				  the index will be put in *entry_loc.

INPUT           : UINT8*       data: data to check.
				  int* entry_index: found index will be put here.

OUTPUT          : None

RETURN          : OK	- suceeded, and found index will be put in *entry_loc
				  ERROR - mac not in the table.
****************************************************************************/
LOCAL STATUS _ruleTbl_checkData( UINT8* data, int* entry_loc)
{
    int ruleNum, grpNum, result;

    if( FILTER_rule_table.currt_tbl_size<=0 )
        return OK;      /* forward if no entry in the table */
	else
	{
        for ( ruleNum = 0, grpNum = 0; grpNum < FILTER_rule_table.currt_tbl_size; ruleNum++)
        {

			if (FILTER_rule_table.filterRuleTblEntry[ruleNum].filterLength != 0)
          	{
				if( _rule_compare( data+FILTER_rule_table.filterRuleTblEntry[ruleNum].filterOffset,
                                	FILTER_rule_table.filterRuleTblEntry[ruleNum].filterData,
                                	FILTER_rule_table.filterRuleTblEntry[ruleNum].filterLength,
                                	FILTER_rule_table.filterRuleTblEntry[ruleNum].filterMask)==0 )
            	{
                	if (FILTER_rule_table.filterRuleTblEntry[ruleNum].filterAction != FILTER_NEXT)
                	{
                        if (FILTER_rule_table.filterRuleTblEntry[ruleNum].filterAction == FILTER_BLOCK)
                            	return ERROR;       /* Drop data */
                    	else    return OK;          /* Forward data */

                	}
            	}
            	else    /* Not match, goto next group */
            	{
                	grpNum++;
                    ruleNum = grpNum * GROUP_SIZE-1;
            	}
  			}
		}
        return OK;
    }

} /* end of _ruleTbl_checkData() */




/****************************************************************************
FUNCTION NAME   : LOCAL int _rule_compare( UINT8* target_data, UINT8* key_rule,
                                           UINT8 length, UINT8 *mask)

DESCRIPTION     : This routine will compare 2 set of data with mask.

INPUT           : UINT8* target data:
                  UINT8*    key_rule:
                  UINT8     length:
                  UINT8*    mask:

OUTPUT          : None

RETURN          : 1   - target_data > key_rule
                 -1   - target_data < key_rule
                  0   - target_data = key_rule
****************************************************************************/
LOCAL int _rule_compare( UINT8* target_data, UINT8* key_rule, UINT8 length, UINT8* mask)
{
	UINT8	i;

    for( i = 0; i < length; i++ )
	{
        if( (*(target_data+i) ^ *(key_rule+i)) & *(mask+i) )
        {
            if( *(target_data+i) > *(key_rule+i) )
                    return 1;
            else    return -1;
        }
    }

	return 0;

} /* end of _rule_compare() */



/****************************************************************************
FUNCTION NAME   : void d_show_filter_tbl(void)

DESCRIPTION     : This routine will print global variables(to console):
                    max_tbl_size,
                    currt_tbl_size,
                  and contents of FILTER_addr_table...
				  For debugging purpose only...

INPUT           : None

OUTPUT          : None

RETURN          : None
****************************************************************************/
void d_show_filter_tbl(void)
{
	int 	i=0;
	int 	j=0;
	char 	sec[70];


	/*----- DEBUG -----*/
	printf("\n\r\n\r");

	printf("filter_function=%d\n\r",FILTER_rule_table.filter_function );
	printf("max_tbl_size=%d\n\r", FILTER_rule_table.max_tbl_size);

	printf("currt_tbl_size=%d\n\r",FILTER_rule_table.currt_tbl_size);

	printf("********** %s **********\n", "Contents of FILTER_rule_table");

	printf("   |        mac        | \n\r" );

    for( i=0; i<FILTER_rule_table.currt_tbl_size; i++)
	{
        printf("#%d   %x %x %x %x %x %x      %x %x %x %x %x %x\n",i,
                    FILTER_rule_table.filterRuleTblEntry[i].filterData[0],
                    FILTER_rule_table.filterRuleTblEntry[i].filterData[1],
                    FILTER_rule_table.filterRuleTblEntry[i].filterData[2],
                    FILTER_rule_table.filterRuleTblEntry[i].filterData[3],
                    FILTER_rule_table.filterRuleTblEntry[i].filterData[4],
                    FILTER_rule_table.filterRuleTblEntry[i].filterData[5],
                    FILTER_rule_table.filterRuleTblEntry[i].filterMask[0],
                    FILTER_rule_table.filterRuleTblEntry[i].filterMask[1],
                    FILTER_rule_table.filterRuleTblEntry[i].filterMask[2],
                    FILTER_rule_table.filterRuleTblEntry[i].filterMask[3],
                    FILTER_rule_table.filterRuleTblEntry[i].filterMask[4],
                    FILTER_rule_table.filterRuleTblEntry[i].filterMask[5]
                    );
	}


} /* end of void d_show_filter_tbl() */



void filter_on(void)
{
    brdgFILTER_CTRL(FILTER_FUNC_ENABLE, 0, 0, NULL);
}



void filter_off(void)
{
    brdgFILTER_CTRL(FILTER_FUNC_DISABLE, 0, 0, NULL);
}


void filter_add(void)
{
    STRUCT_BrdgFilterRuleTblEntry entry;

    memmove(&entry, "\x00\x00\xc0\x2e\x02\xda\x00\x00\xc0\x2e\x02\xda\x00\x00\xc0\x2e\x02", 17 );

    brdgFILTER_CTRL(FILTER_TBL_ADD_ENTRY, 1, 1, (void*) &entry);
}


void filter_delete(void)
{
 	int pidx = 1;
	brdgFILTER_CTRL(FILTER_TBL_DEL_ENTRY, 1, 1, &pidx);
}



int filter_size(void)
{
    return (FILTER_rule_table.currt_tbl_size * GROUP_SIZE);
}

#if (1)

BOOL
BridgeFilterLoadRuleCfg
(
	char*							pRegPath,
	STRUCT_BrdgFilterRuleTblEntry*	pCfg
)
{
	BOOL				bResult = FALSE;

	JX_REG_KEY_HANDLE	hKey	= NULL;

	if ((hKey = JxRegOpenKey(NULL, pRegPath)) == NULL)
	{
		/* Error! */
	}
	else
	{
		DWORD	Enabled	= 0;
		DWORD	Offset	= 0;
		DWORD	Last	= 0;
		DWORD	Forward	= 0;

		JX_REG_VAL_DESC StructDesc[]	=
			{
				{"Enabled", JX_REG_VAL_DWORD, sizeof(Enabled), &Enabled},
				{"Offset", JX_REG_VAL_DWORD, sizeof(Offset), &Offset},
				{"Mask", JX_REG_VAL_BINARY, sizeof(pCfg->filterMask), &pCfg->filterMask},
				{"Data", JX_REG_VAL_BINARY, sizeof(pCfg->filterData), &pCfg->filterData},
				{"Last", JX_REG_VAL_DWORD, sizeof(Last), &Last},
				{"Forward", JX_REG_VAL_DWORD, sizeof(Forward), &Forward}
			};

		if (!JxRegQueryStruct(hKey, NUMBER_OF_ELEMENTS(StructDesc), StructDesc))
		{
			/* Error! */
		}
		else
		{
			JX_REG_VAL_HANDLE	hValue	= NULL;
			JX_REG_VAL_TYPE		Type	= 0;
			size_t				Size	= 0;

			hValue	= JxRegOpenValue(hKey, "Mask");

			JxRegGetValueInfo(hValue, &Type, &Size);

			pCfg->filterLength = (UINT8)Size;

			JxRegCloseValue(hValue);

			pCfg->filterActive = Enabled;
			pCfg->filterOffset = Offset;
			pCfg->filterNext = !Last;
			pCfg->filterAction = Forward;

			bResult = TRUE;
		}

		JxRegCloseKey(hKey);
	}

	return bResult;
}

BOOL
BridgeFilterSaveRuleCfg
(
	char*							pRegPath,
	STRUCT_BrdgFilterRuleTblEntry*	pCfg
)
{
	BOOL				bResult = FALSE;

	JX_REG_KEY_HANDLE	hKey	= NULL;

	if ((hKey = JxRegCreateKey(NULL, pRegPath)) == NULL)
	{
		/* Error! */
	}
	else
	{
		DWORD	Enabled	= pCfg->filterActive;
		DWORD	Offset	= pCfg->filterOffset;
		DWORD	Last	= !pCfg->filterNext;
		DWORD	Forward	= pCfg->filterAction;

		JX_REG_VAL_DESC StructDesc[]	=
			{
				{"Enabled", JX_REG_VAL_DWORD, sizeof(Enabled), &Enabled},
				{"Offset", JX_REG_VAL_DWORD, sizeof(Offset), &Offset},
				{"Mask", JX_REG_VAL_BINARY, pCfg->filterLength, &pCfg->filterMask},
				{"Data", JX_REG_VAL_BINARY, pCfg->filterLength, &pCfg->filterData},
				{"Last", JX_REG_VAL_DWORD, sizeof(Last), &Last},
				{"Forward", JX_REG_VAL_DWORD, sizeof(Forward), &Forward}
			};

		if (!JxRegSetStruct(hKey, NUMBER_OF_ELEMENTS(StructDesc), StructDesc))
		{
			/* Error! */
		}
		else
		{
			bResult = TRUE;
		}

		JxRegCloseKey(hKey);
	}

	return bResult;
}

BOOL
BridgeFilterLoadRuleListCfg
(
	char*							pRegPath,
	STRUCT_BrdgFilterRuleTbl*	pCfg
)
{
	BOOL				bResult	= FALSE;

	BOOL				bError	= FALSE;

	JX_REG_KEY_HANDLE	hKey	= NULL;

	if ((hKey = JxRegOpenKey(NULL, pRegPath)) == NULL)
	{
		/* Error! */

		bError = TRUE;
	}
	else
	{
		JX_REG_KEY_HANDLE	hEntryKey	= NULL;

		ULONG				i;

		for
		(
			i = 0;
			(
				i < NUMBER_OF_ELEMENTS(pCfg->filterRuleTblEntry) &&
				((hEntryKey = JxRegEnumerateKey(hKey, hEntryKey)) != NULL)
			);
			i++
		)
		{
			char	EntryPath[JX_REG_PATH_SIZE_MAX]	= "";

			sprintf(EntryPath, "%s\\%s", pRegPath, JxRegGetKeyInfo(hEntryKey));

			bError |= !BridgeFilterLoadRuleCfg
				(
					EntryPath,
					&pCfg->filterRuleTblEntry[i]
				);

			JxRegCloseKey(hEntryKey);
		}

		if (!bError)
		{
			pCfg->currt_tbl_size = (i + GROUP_SIZE - 1) / GROUP_SIZE;
		}

		JxRegCloseKey(hKey);
	}

	bResult = !bError;

	return bResult;
}

BOOL
BridgeFilterSaveRuleListCfg
(
	char*							pRegPath,
	STRUCT_BrdgFilterRuleTbl*	pCfg
)
{
	BOOL	bResult		= FALSE;

	BOOL	bError		= FALSE;

	JX_REG_KEY_HANDLE	hKey	= NULL;

	if ((hKey = JxRegCreateKey(NULL, pRegPath)) == NULL)
	{
		/* Error! */

		bError = TRUE;
	}
	else
	{
		ULONG	i;

		for (i = 0; i < (pCfg->currt_tbl_size * GROUP_SIZE); i++)
		{
			char	EntryPath[JX_REG_PATH_SIZE_MAX]	= "";

			sprintf(EntryPath, "%s\\%04u", pRegPath, i);

			bError |= !BridgeFilterSaveRuleCfg(EntryPath, &pCfg->filterRuleTblEntry[i]);
		}
	}

	bResult = !bError;

	return bResult;
}

BOOL
BridgeFilterLoad
(
	char*	pRegPath
)
{
	BOOL				bResult = FALSE;

	BOOL				bError	= FALSE;

	JX_REG_KEY_HANDLE	hKey	= NULL;

	STRUCT_BrdgFilterRuleTbl*	pCfg	= &FILTER_rule_table;

	memset(pCfg, 0, sizeof(*pCfg));

	if ((hKey = JxRegOpenKey(NULL, pRegPath)) == NULL)
	{
		/* Error! */

		bError = TRUE;
	}
	else
	{
		DWORD	Enabled	= 0;

		JX_REG_VAL_DESC StructDesc[]	=
			{
				{"Enabled", JX_REG_VAL_DWORD, sizeof(Enabled), &Enabled}
			};

		if (!JxRegQueryStruct(hKey, NUMBER_OF_ELEMENTS(StructDesc), StructDesc))
		{
			/* Error! */

			bError = TRUE;
		}
		else
		{
			char	EntryPath[JX_REG_PATH_SIZE_MAX]	= "";

			sprintf(EntryPath, "%s\\Rule", pRegPath);

			pCfg->filter_function = Enabled;

			bError |= !BridgeFilterLoadRuleListCfg
				(
					EntryPath,
					pCfg
				);

		}

		JxRegCloseKey(hKey);
	}

	bResult = !bError;

 	/* filter_function to global brdg_filter_is_on(used by protocol.c */    
	brdg_filter_is_on=pCfg->filter_function;

	return bResult;
}

BOOL
BridgeFilterSave
(
	char*	pRegPath
)
{
	BOOL				bResult = FALSE;

	BOOL				bError	= FALSE;

	JX_REG_KEY_HANDLE	hKey	= NULL;

	STRUCT_BrdgFilterRuleTbl*	pCfg	=
		&FILTER_rule_table;

	if ((hKey = JxRegCreateKey(NULL, pRegPath)) == NULL)
	{
		/* Error! */

		bError = TRUE;
	}
	else
	{
		DWORD	Enabled	= pCfg->filter_function;

		JX_REG_VAL_DESC StructDesc[]	=
			{
				{"Enabled", JX_REG_VAL_DWORD, sizeof(Enabled), &Enabled}
			};

		if (!JxRegSetStruct(hKey, NUMBER_OF_ELEMENTS(StructDesc), StructDesc))
		{
			/* Error! */

			bError = TRUE;
		}
		else
		{
			char	EntryPath[JX_REG_PATH_SIZE_MAX]	= "";

			sprintf(EntryPath, "%s\\Rule", pRegPath);

			bError |= !BridgeFilterSaveRuleListCfg
				(
					EntryPath,
					pCfg
				);

		}

		JxRegCloseKey(hKey);
	}

	bResult = !bError;

	return bResult;
}

BOOL
BridgeFilterCreateDefault
(
)
{
	BOOL		bResult = FALSE;

	STRUCT_BrdgFilterRuleTbl*	pCfg	= &FILTER_rule_table;

	STRUCT_BrdgFilterRuleTbl	Default	=
	{
		FALSE,
		0,		/* TBD: Remove. */
		0,
		{0, 0, 0, {0}, {0}, FALSE, FALSE}
	};

	memset(pCfg, 0, sizeof(*pCfg));

	*pCfg = Default;

	bResult = TRUE;

	return bResult;
}

#endif /* (0) */







/*
 * Convert hex digit to hex character and vice-versa
 */

/* Hex char string to hex digit array */
int parseStr2Hex(char *str, UINT8* data, int hexcnt)
{
    char *s;
    int i, c1, c2;

    /* Start at beginning */
    s = str + strlen(str);
    //hexcnt = strlen(str)/2 + 1;
    for (i = hexcnt-1; (i >= 0) && (s >= str); i--)
    {
	c2 = (s > str && isxdigit((int) s[-1])) ? xdigit2i((int) *--s) : 0;
	c1 = (s > str && isxdigit((int) s[-1])) ? xdigit2i((int) *--s) : 0;
    *(data+i) = c1 * 16 + c2;
    }
    return((s <= str+1) ? 0 : -1);
}


/* Hex digit array to hex char string */
void formatHex2Str(char* buf, char* str, int len)
{
    int i;

    for (i = 0; i < len; i++)
        {
        *buf++ = i2xdigit(*(str+i) >> 4);
        *buf++ = i2xdigit(*(str+i));
        }
    *buf = NULL;
}



char i2xdigit(char digit)
{
    digit &= 0xf;
    return (digit > 9) ? digit - 10 + 'a' : digit + '0';
}

char xdigit2i(char digit)
{
    if (digit >= '0' && digit <= '9') return (digit - '0'     );
    if (digit >= 'a' && digit <= 'f') return (digit - 'a' + 10);
    if (digit >= 'A' && digit <= 'F') return (digit - 'A' + 10);
    return 0;
}

/***************************************************************************/
/* $History$
 */
