/****************************************************************************
****    Copyright (c) 2002
****    Conexant Systems Inc. (formerly Rockwell Semiconductor Systems)
****    Personal Computing Division
****    All Rights Reserved
****
****    CONFIDENTIAL and PROPRIETARY --
****        No Dissemination or use without prior written permission.
****
*****************************************************************************
MODULE NAME:
        Network Address Port Translation

FILE NAME:
        alias_netmeeting.c

ABSTRACT:

DETAILS:

KEYWORDS:
        $Archive:   P:/d874/hasbani/archives/NAT_freebsd/alias_netmeeting.c_v  $
        $Revision: 1.1.1.1 $
        $Date: 2004/04/01 09:10:01 $
****************************************************************************/

/***************************************************************************/
/*                    INCLUDE SECTION                                      */
/***************************************************************************/
#include <vxWorks.h>
#include <stdlib.h>
#include <bspcfg.h>
/* BSD network include files */
#include <netinet/in_systm.h>
#include <netLib.h>
#include <inetLib.h>
#include <net/mbuf.h>   /* struct mbuf and struct ifnet definitions */
#include <netinet/in.h>
#include <netinet/ip.h>
#include <netinet/tcp.h>
#include <netinet/udp.h>

#include "protoutils.h"
#include "cfgmgr/JxRegSrv.h"
#include "cfgmgr/JxCfgMgr.h"
#include "cfgmgr/flash_api.h"

#include <alias.h>
#include <alias_local.h>
#include <app_conf/h/system_cfg.h>

#ifdef SYSLOG_ENABLED
#include <syslog/h/logdrv.h>    /* Include SYSLOG utility */
#endif  /* SYSLOG_ENABLED */

/***************************************************************************/
/*                LOCAL CONSTANT DEFINITION SECTION                        */
/***************************************************************************/
#undef NAT_H323_DEBUG    /* Turn on NAT H323 debug message */

/* NetMeeting */
#define H225_CALL_SIGNAL_PORT_NUMBER 1720    /* H225 */
#define T120_PORT_NUMBER             1503

#define MONITOR_LIMIT_DEFAULT 16
/***************************************************************************
 *                      LOCAL TYPEDEF                                      *
 ***************************************************************************/

/***************************************************************************/
/*              LOCAL MACRO SECTION                                        */
/***************************************************************************/
#ifdef NAT_H323_DEBUG
#ifdef SHOWTIME
#define NAT_H323_printf(x)   WriteFIFOString(x, strlen(x), 0)
#else
/* No SHOWTIME Code */
#define NAT_H323_printf(x)   printf(x)
#endif  /* SHOWTIME */
#else
/* NAT_H323_DEBUG is turned off */
#define NAT_H323_printf(x)   Print(x)
#endif  /*NAT_H323_DEBUG */

#define ARRAYSIZE(X) (sizeof(X)/sizeof(X[0]))


/***************************************************************************/
/*              LOCAL TYPE DEFINITION                                      */
/***************************************************************************/
typedef enum _state { 
    UNUSED, 
    OPENED, /* conversation has started but channel opens aren't expected */
    NEXT_CHANNEL_MATTERS, /* next channel open needs to be monitored */
    DONE, /* not yet used */
    SECONDARY_TCP_CONNECTION /* this state needs constant monitoring */
} State;

typedef struct _conversation {
    State state;
} Conversation;

/* singly linked list of ports currently being watched */
typedef struct _port_list_entry 
{
    u_short port;
    struct alias_link *link;
    struct _port_list_entry *next;
} PortListEntry;


/***************************************************************************/
/*              LOCAL VARIABLES DECLARATION                                */
/***************************************************************************/
LOCAL char printBuffer[80];
LOCAL PortListEntry *ports = 0;
LOCAL PortListEntry *unloadqueue = 0;
LOCAL int monitor_limit = MONITOR_LIMIT_DEFAULT;
LOCAL BOOL netmeeting_initialized = FALSE;
LOCAL int NetMeetingDebugTaskId = 0;

/***************************************************************************/
/*              EXTERNAL VARIABLES DECLARATION                             */
/***************************************************************************/


/***************************************************************************/
/*              LOCAL FUNCTION PROTOTYPE SECTION                           */
/***************************************************************************/
LOCAL Conversation *new_connection( void );
/*
   adds a new portnumber to the list of ports being monitored by this code 
*/
static STATUS netmeeting_monitor_port(u_short portno);
static int netmeeting_queue_port(u_short portno);
static int netmeeting_unregister_list(PortListEntry **cptr);
static BOOL search_netmeeting_monitor_port(u_short portno);


/***************************************************************************/
/*              GLOBAL FUNCTION PROTOTYPE SECTION                          */
/***************************************************************************/
STATUS NatNetMeetingInit(void)
{
    int i;
    u_short init_ports[] = { H225_CALL_SIGNAL_PORT_NUMBER };
    STATUS status = ERROR;

#if (defined(NAT_H323_DEBUG) && defined(SYSLOG_ENABLED))
    if (monitor_limit != MONITOR_LIMIT_DEFAULT) {
        SYS_Log(LOG_INFO, "NETMEETING: monitor_limit=%d", 
                    monitor_limit, 0, 0, 0, 0, 0);
    }
#endif  /* defined(NAT_H323_DEBUG) && defined(SYSLOG_ENABLED) */

    if (netmeeting_initialized == TRUE)
    {
        netmeeting_unregister_list(&ports);
    }

    for (i=0; i < ARRAYSIZE(init_ports); i++) {
        if (init_ports[i]) {
            status = netmeeting_monitor_port(init_ports[i]);
            if (status == ERROR) 
                return status;
        } 
    }
    netmeeting_initialized = TRUE;
    return status;
}   /* End of NatNetMeetingInit() */

void AliasHandleNetMeetingOut(struct ip *iph, struct alias_link *link)
{
    int             hlen, tlen, dlen;
    unsigned char   *sptr;
    struct tcphdr   *tcph;
    unsigned char   *data;
    unsigned char   *data_limit;
    Conversation     *conn = (Conversation *)GetNetMeetingState(link);

    if (netmeeting_initialized == FALSE)
    {
        return;
    }
    netmeeting_unregister_list(&unloadqueue);

    /* Calculate data length of TCP packet */
    tcph = (struct tcphdr *) ((char *) iph + (iph->ip_hl << 2));
    hlen = (iph->ip_hl + tcph->th_off) << 2;

    if (
        search_netmeeting_monitor_port(ntohs(tcph->th_dport)) ||
        search_netmeeting_monitor_port(ntohs(tcph->th_sport))
        )
    {
#if (defined(NAT_H323_DEBUG))
        sprintf(printBuffer, "NAT_NETMEETING: Monitoring on port %d --> %d\n\r",
            ntohs(tcph->th_sport), ntohs(tcph->th_dport));
        NAT_H323_printf(printBuffer);
#endif  /* NAT_H323_DEBUG */
    }
    else
    {
#if (defined(NAT_H323_DEBUG))
        if (ntohs(tcph->th_dport) == T120_PORT_NUMBER ||
            ntohs(tcph->th_sport) == T120_PORT_NUMBER)
        {

        }
        else
        {
            sprintf(printBuffer, "*****NAT_NETMEETING: monitoring on port %d --> %d*****\n\r",
                ntohs(tcph->th_sport), ntohs(tcph->th_dport));
            NAT_H323_printf(printBuffer);
        }
#endif  /* NAT_H323_DEBUG */
        return;
    }
    tlen = iph->ip_len;

    dlen = tlen - hlen;

    /* Place string pointer and beginning of data */
    sptr = (char *) iph;
    sptr += hlen;

    data = sptr;
    data_limit = (char *)sptr + dlen;

    /* avoid overrun in loop below */
    data_limit -= 3;

    if (tcph->th_ack == 0) 
    {
        if (GetNetMeetingState(link) == NULL) 
        { 
            SetNetMeetingState(link, (void *)new_connection());
            conn = (Conversation *)GetNetMeetingState(link);
        }

        if (ntohs(tcph->th_dport) != H225_CALL_SIGNAL_PORT_NUMBER) 
        {
            conn->state = SECONDARY_TCP_CONNECTION;
#if (defined(NAT_H323_DEBUG))
            sprintf(printBuffer, "NAT_NETMEETING: monitoring secondary TCP connection on port %d --> %d\n\r",
                ntohs(tcph->th_sport), ntohs(tcph->th_dport));
            NAT_H323_printf(printBuffer);
#endif  /* defined(NAT_H323_DEBUG) */
        }
#if (defined(NAT_H323_DEBUG))
        else 
        {
            char output[INET_ADDR_LEN];

            NAT_H323_printf("primary netmeeting conn\n\r");

            inet_ntoa_b(GetOriginalAddress(link), output);
            sprintf(printBuffer, "   caller: %s: %d\n\r",
                output, ntohs(GetOriginalPort(link)));
            NAT_H323_printf(printBuffer);

            inet_ntoa_b(GetAliasAddress(link), output);
            sprintf(printBuffer, "   firewall: %s: %d\n\r", 
                output, ntohs(GetAliasPort(link)));
            NAT_H323_printf(printBuffer);

            inet_ntoa_b(GetDestAddress(link), output);
            sprintf(printBuffer, "   callee: %s: %d\n\r",
                output, ntohs(GetDestPort(link)));
            NAT_H323_printf(printBuffer);
        }
#endif  /* defined(NAT_H323_DEBUG) */
    }

    if ((tcph->th_flags & TH_FIN) && ntohs(GetOriginalPort(link)) != H225_CALL_SIGNAL_PORT_NUMBER) {
#if (defined(NAT_H323_DEBUG))
        sprintf(printBuffer, "fin -- should shortly unload support on %d\n\r",
                ntohs(GetOriginalPort(link)));
        NAT_H323_printf(printBuffer);
#endif  /* defined(NAT_H323_DEBUG) */
    }

    if (!conn) {
        return;
    }

#if (defined(NAT_H323_DEBUG))
    sprintf(printBuffer, "Analyzing %d --> %d packet\n\r",
                ntohs(tcph->th_sport), ntohs(tcph->th_dport));
    NAT_H323_printf(printBuffer);
#endif  /* defined(NAT_H323_DEBUG) */

    while (data < data_limit)
    {
        struct in_addr temp_ip;
        u_short temp_port;

        temp_ip = GetDestAddress(link);
        if (conn->state == NEXT_CHANNEL_MATTERS &&
            !memcmp(data, &(temp_ip.s_addr), 4)) 
        {
            STATUS ret = ERROR; 
            u_short port = (((UINT16)(*(data+4)) )<< 8) | (UINT16)(*(data+5)); 

#if (defined(NAT_H323_DEBUG))
            u_long ip = ((*(data) << 24) + (*(data + 1) << 16) +
                                    (*(data + 2) << 8) + (*(data + 3)));
            char output[INET_ADDR_LEN];
            struct in_addr  ip_addr;
        
            ip_addr.s_addr = htonl(ip);
            inet_ntoa_b(ip_addr, output);
            sprintf(printBuffer, "Callee's IP Address: %s\n\r", output);
            NAT_H323_printf(printBuffer);

            sprintf(printBuffer, "channel from in-->out opened on 0x%04x\n\r", port);
            NAT_H323_printf(printBuffer);
#endif  /* defined(NAT_H323_DEBUG) */

            ret = netmeeting_monitor_port(port);

#if (defined(NAT_H323_DEBUG))
            sprintf(printBuffer, "monitor ret: %s\n\r", (ret == OK? "OK": "ERROR"));
            NAT_H323_printf(printBuffer);
#endif  /* defined(NAT_H323_DEBUG) */
        }

        temp_ip = GetDestAddress(link);
        if (conn->state == OPENED && !memcmp(data, &(temp_ip.s_addr), 4)) 
        {

#if (defined(NAT_H323_DEBUG))
            NAT_H323_printf("next channel open matters.\n\r");
#endif  /* defined(NAT_H323_DEBUG) */

            conn->state = NEXT_CHANNEL_MATTERS;
        }
        else
        {
            temp_ip = GetAliasAddress(link);
            if (conn->state == OPENED && 
                   !memcmp(data, &(temp_ip.s_addr), 4)) 
            {
#if (defined(NAT_H323_DEBUG))
            NAT_H323_printf("someone is calling in; next ch open matters; rewrite.\n\r");
#endif  /* defined(NAT_H323_DEBUG) */

            conn->state = NEXT_CHANNEL_MATTERS;
            temp_ip = GetOriginalAddress(link);
            memcpy(data, &(temp_ip.s_addr), 4);
            
            /* Compute TCP checksum for revised packet */
            tcph->th_sum = 0;
            tcph->th_sum = TcpChecksum(iph);
            data++;
            continue;
            }
        }

        temp_ip = GetOriginalAddress(link);
        if (!memcmp(data, &(temp_ip.s_addr), 4))
        {
            /* our IP address */
            u_short port;
            struct in_addr  alias_address;

#if (defined(NAT_H323_DEBUG))
            u_long ip = ((*(data) << 24) + (*(data + 1) << 16) + (*(data + 2) << 8) + (*(data + 3)));
            struct in_addr  ip_addr;
            char    output[INET_ADDR_LEN];

            ip_addr.s_addr = htonl(ip);
            inet_ntoa_b(ip_addr, output);
            sprintf(printBuffer, "Caller's IP Address: %s\n\r", output);
            NAT_H323_printf(printBuffer);

            NAT_H323_printf("rewriting IP\n\r");
#endif  /* NAT_H323_DEBUG */
            alias_address = GetAliasAddress(link);
            memcpy(data, (char *)&(alias_address), 4);

           /* Compute TCP checksum for revised packet */
            tcph->th_sum = 0;
            tcph->th_sum = TcpChecksum(iph);
            data += 4;

            port = (((UINT16)(*data))<<8) | ((UINT16)(*(data+1)));
            if (conn->state == SECONDARY_TCP_CONNECTION && 
                           port != T120_PORT_NUMBER) 
            {
                struct alias_link *pLink = NULL;
                    
                pLink = FindUdpTcpOut(GetOriginalAddress(link),
                                        GetDestAddress(link),
                                        htons(port), 0, IPPROTO_UDP, 0);
                if (pLink == NULL)
                {
                    pLink = FindUdpTcpOut(GetOriginalAddress(link),
                                          GetDestAddress(link), htons(port), 0,
                                          IPPROTO_UDP, 1);
                }
                else
                {
                    /* reset expiry to keep entry alive */
                }
                /* pLink->alias_port contains the port on which data
                   must arrive to be correctly forwarded */
                temp_port = GetAliasPort(pLink);
                memcpy(data, &(temp_port), 2);
                
                /* Compute TCP checksum for revised packet */
                tcph->th_sum = 0;
                tcph->th_sum = TcpChecksum(iph);
#if (defined(NAT_H323_DEBUG))
                {
                    int p = (((int)(*data))<<8)|((int)(*(data+1)));
                    sprintf(printBuffer, "REWROTE port %x --> %x\n\r", 
                                port, p);
                    NAT_H323_printf(printBuffer);
                }
                NAT_H323_printf("forwarding UDP data\n\r");
#endif  /* NAT_H323_DEBUG */
            }
            else if (port != T120_PORT_NUMBER) 
            {

                STATUS ret = ERROR;
                struct alias_link *pLink;

                pLink = FindNetMeetingIn(GetOriginalAddress(link), htons(port),
                    GetDestAddress(link), 0, GetAliasAddress(link), htons(port),
                    IPPROTO_TCP);

                if (pLink != NULL)
                {
                    /* pLink->alis_port contains the port on which data
                       must arrive to be correctly forwarded */
                    temp_port = GetAliasPort(pLink);
                    memcpy(data, &(temp_port), 2);

                    ret = netmeeting_monitor_port(port);
#if (defined(NAT_H323_DEBUG))
                    sprintf(printBuffer, "monitor on %d return: %s\n\r", port, 
                                                (ret == OK ? "OK" : "ERROR"));
                    NAT_H323_printf(printBuffer);
#endif  /* defined(NAT_H323_DEBUG) */

                    /* Compute TCP checksum for revised packet */
                    tcph->th_sum = 0;
                    tcph->th_sum = TcpChecksum(iph);
#if (defined(NAT_H323_DEBUG))
                    {
                        int p = (((int)(*data))<<8)|((int)(*(data+1)));
                        sprintf(printBuffer, "REWROTE port %x --> %x\n\r", 
                                port, p);
                        NAT_H323_printf(printBuffer);
                    }
                    NAT_H323_printf("forwarding TCP data\n\r");
#endif  /* NAT_H323_DEBUG */
                }
                else
                {
#ifdef NAT_H323_DEBUG
                    NAT_H323_printf("failed in create an incoming entry\n\r");
#endif  /* NAT_H323_DEBUG */
                }
            }
        }
        data++;
    }

#if (defined(NAT_H323_DEBUG) )
    NAT_H323_printf("Done processing packet\n\r");
#endif  /* NAT_H323_DEBUG */

    return;
}   /* End of AliasHandleNetMeetingOut() */

void NAT_netmeeting_queue_port(u_short portno)
{
    netmeeting_queue_port(portno);
}   /* End of NAT_netmeeting_queue_port() */

/*
   new_connection is used to allocate the connection structure, which is 
   placed in the app_data portion of the masq record for each connection.
*/
LOCAL Conversation *new_connection( void )
{
    Conversation *ret = malloc(sizeof(Conversation));

    if (ret == NULL) 
        return ret;
    ret->state = OPENED;

    return ret;
}

static STATUS netmeeting_monitor_port(u_short portno)
{
    int ret = 0;
    PortListEntry *newmasq;
    int limit_check = monitor_limit;

    PortListEntry *counter = ports;

    while (counter) 
    {
        if (counter->port == portno)
        {
            return OK;
        }
        limit_check--;
        counter = counter->next;
    }

    if (limit_check <= 0) 
    {
#ifdef SYSLOG_ENABLED
        SYS_Log(LOG_INFO, "NETMEETING: port limit exceeded", 0, 0, 0, 0, 0, 0);
#endif
        return ERROR;
    }

    newmasq = malloc(sizeof(*newmasq));
    if (newmasq == NULL) 
    {
#ifdef SYSLOG_ENABLED
        SYS_Log(LOG_INFO, "IP_MASQ_NETMEETING: ERROR out of RAM!", 0, 0, 0, 0, 0, 0);
#endif  /* SYSLOG_ENABLED */
        return ERROR;
    }

    newmasq->port = portno;
    newmasq->next = ports;

#if (defined(NAT_H323_DEBUG))
    sprintf(printBuffer, "NETMEETING: monitoring port %d\n\r", portno);
    NAT_H323_printf(printBuffer);
#endif  /* NAT_H323_DEBUG */

    ports = newmasq;
    return OK;
}

static int netmeeting_queue_port(u_short portno)
{
    int ret = 0;
    PortListEntry *predecessor;
    PortListEntry *removeme;

    if (portno == H225_CALL_SIGNAL_PORT_NUMBER)
    {
        return ret;
    }

    predecessor = 0;
    removeme = ports;
    while (removeme != 0 && removeme->port != portno) 
    {
        predecessor = removeme;
        removeme = removeme->next;
    }
    if (removeme == 0) 
    {
        /* no matching port found? bail out */
#if (defined(NAT_H323_DEBUG))
        sprintf(printBuffer, "NETMEETING: port %d not being monitored?\n\r", portno);
        NAT_H323_printf(printBuffer);
#endif  /* NAT_H323_DEBUG */

        return -1;
    }

    if (predecessor == 0) 
    {
        if (removeme != ports) 
        {
            /* impossible! */
#if (defined(NAT_H323_DEBUG))
            NAT_H323_printf("NETMEETING: impossible?!\n\r");
#endif  /* NAT_H323_DEBUG */
            return -1;
        }
        ports = ports->next;
    }
    else 
    {
        if (removeme != predecessor->next) 
        {
            /* impossible! */
#if (defined(NAT_H323_DEBUG))
            NAT_H323_printf("NETMEETING: impossible?!!\n\r");
#endif  /* NAT_H323_DEBUG */
            return -1;
        }
        predecessor->next = removeme->next;
    }

#if (defined(NAT_H323_DEBUG))
    sprintf(printBuffer, "NETMEETING: queued remove on %d\n\r", removeme->port);
    NAT_H323_printf(printBuffer);
#endif  /* NAT_H323_DEBUG */

    removeme->next = unloadqueue;
    unloadqueue = removeme;

    return ret;
}

static int netmeeting_unregister_list(PortListEntry **cptr)
{
    int tmp, ret;
    PortListEntry *freeme;
    PortListEntry *cur;

    if (!cptr) return -1;
    cur = (*cptr);
    (*cptr) = 0;

    ret = 0;
    while (cur != 0)
    {
        freeme = cur;
        cur = cur->next;
        free(freeme);
        freeme = 0;
    }
    return ret;
}   /* End of netmeeting_unregister_list() */

static BOOL search_netmeeting_monitor_port(u_short portno)
{
    BOOL            found = FALSE;
    PortListEntry *counter = ports;

    while (counter && (found == FALSE) ) 
    {
        if (counter->port == portno)
        {
            found = TRUE;
        }
        else
        {
            counter = counter->next;
        }
    }

    return (found);
}   /* End of search_netmeeting_monitor_port() */

LOCAL void DebugNetMeetingMain(int delay)
{
    PortListEntry *counter = NULL;
    FOREVER
    {
        taskDelay(delay * sysClkRateGet()); 
        counter = ports;
        while (counter) 
        {
#if (defined(NAT_H323_DEBUG))
            sprintf(printBuffer, "DebugNetMeeting: port number: %d\n\r", counter->port);
            NAT_H323_printf(printBuffer);
#endif  /* NAT_H323_DEBUG */
            counter = counter ->next;
        }

    }
}   /* End of NatDebugTaskMain() */


STATUS NetMeetingDebugTaskStart(int delay)
{
    int delay_time = 0;

    if (delay == 0)
    {
        delay_time = 10;    /* This is the default delay period */
    }

    if (NetMeetingDebugTaskId == 0)
    {
        if ( (NetMeetingDebugTaskId = taskSpawn("tDebugH323",
                                    200,    /* task priority */
                                    0,      /* task option */
                                    0x1000, /* task stack size */
                                    (FUNCPTR)DebugNetMeetingMain,
                                    delay_time,0,0,0,0,0,0,0,0,0))
                            == ERROR)
        {
            return ERROR;
        }
        else
        {
            return OK;
        }
    }
    return ERROR;
}   /*  End of NetMeetingDebugTaskStart() */