/* dhcpRelay.c - DHCP server and relay agent shared code library */

/* Copyright 1984 - 1997 Wind River Systems, Inc. */
#include "copyright_wrs.h"

/*
modification history
____________________
01f,04dec97,spm  added code review modifications
01e,06oct97,spm  split interface name into device name and unit number; fixed
                 errors in debugging output
01d,02jun97,spm  changed DHCP option tags to prevent name conflicts (SPR #8667)
                 and updated man pages
01c,28apr97,spm  limited maximum number of hops to 16 for RFC compliance
01b,18apr97,spm  added conditional include DHCPR_DEBUG for displayed output
01a,07apr97,spm  created by modifying WIDE project DHCP implementation
*/

/*
DESCRIPTION
This library contains the code used by the DHCP relay agent to transfer 
packets between DHCP or BOOTP clients and DHCP servers. The DHCP server
will also use this code if configured to act as a relay agent.

INCLUDE_FILES: dhcprLib.h
*/

/*
 * WIDE Project DHCP Implementation
 * Copyright (c) 1995 Akihiro Tominaga
 * Copyright (c) 1995 WIDE Project
 * All rights reserved.
 *
 * Permission to use, copy, modify and distribute this software and its
 * documentation is hereby granted, provided only with the following
 * conditions are satisfied:
 *
 * 1. Both the copyright notice and this permission notice appear in
 *    all copies of the software, derivative works or modified versions,
 *    and any portions thereof, and that both notices appear in
 *    supporting documentation.
 * 2. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by WIDE Project and
 *      its contributors.
 * 3. Neither the name of WIDE Project nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE DEVELOPER ``AS IS'' AND WIDE
 * PROJECT DISCLAIMS ANY LIABILITY OF ANY KIND FOR ANY DAMAGES
 * WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE. ALSO, THERE
 * IS NO WARRANTY IMPLIED OR OTHERWISE, NOR IS SUPPORT PROVIDED.
 *
 * Feedback of the results generated from any improvements or
 * extensions made to this software would be much appreciated.
 * Any such feedback should be sent to:
 * 
 *  Akihiro Tominaga
 *  WIDE Project
 *  Keio University, Endo 5322, Kanagawa, Japan
 *  (E-mail: dhcp-dist@wide.ad.jp)
 *
 * WIDE project has the rights to redistribute these changes.
 */

/* includes */

#include "vxWorks.h"
#include "vxLib.h"             /* checksum() declaration. */

#include <stdio.h>
#include <stdlib.h>
#include <sys/ioctl.h>

#include <netinet/in.h>
#include <arpa/inet.h>

#include "etherLib.h"
#include "logLib.h"
#include "sockLib.h"
#include "ioLib.h"

#ifdef DHCPR_DEBUG
#include "inetLib.h"
#endif

#include "dhcprLib.h"
#include "dhcp/dhcp.h"
#include "dhcp/common.h"
#include "dhcp/common_subr.h"

/* globals */

DHCP_TARGET_DESC *	pDhcpRelayTargetTbl;
struct server *		pDhcpTargetList = NULL;
int dhcpNumTargets = 0;
struct msg dhcpMsgIn;
struct msg dhcprMsgOut;
struct msg dhcprFwdMsg;    /* This is used for DHCPR BIPOA forwarding */

/* forward declarations */

void dhcpServerRelay (struct if_info *);
void dhcpClientRelay (struct if_info *, int);
LOCAL STATUS forwarding();

IMPORT void Dhcpr_forwarding(struct msg *, UINT32 ip_addr);
IMPORT BOOL Dhcpr_BIPoA_SubnetBroadcast(void);

IMPORT int  dhcpr_socket_fd;

/*******************************************************************************
*
* dhcpServerRelay - send incoming DHCP/BOOTP message to client port
*
* This routine relays a DHCP or BOOTP message to the client port of 
* every DHCP server or relay agent whose IP address is contained in the 
* circular linked list of targets. That list is constructed during startup of 
* the relay agent (or server, if configured to relay messages). The routine
* accesses global pointers already set to indicate the outgoing message. 
* All messages are discarded after the hops field exceeds 16 to comply with
* the relay agent behavior specified in RFC 1542. The relay agent normally
* discards such messages before this routine is called. They will only
* be received by this routine if the user ignores the instructions in the
* manual and sets the value of DHCP_MAX_HOPS higher than 16.
*
* RETURNS: N/A
*
* ERRNO: N/A
*
* NOMANUAL
*/

void dhcpServerRelay
    (
    struct if_info *ifp
    )
    {
    unsigned int hash;
    int i;
    struct server *srvp;

    dhcpMsgIn.dhcp->hops++;
    if (dhcpMsgIn.dhcp->hops >= 17)    /* RFC limits to 16 hops - ignore. */
        return;

    if (dhcpMsgIn.dhcp->giaddr.s_addr == 0)
        dhcpMsgIn.dhcp->giaddr.s_addr = ifp->ipaddr.s_addr;

    /* Quick and dirty load balancing - pick starting point in circular list. */

    hash = (unsigned)checksum( (u_short *)dhcpMsgIn.dhcp->chaddr, 
                               (MAX_HLEN / 2));

    hash %= dhcpNumTargets;

    srvp = pDhcpTargetList;
    for (i = 0; i < hash; i++)
        srvp = srvp->next;

    forwarding (srvp);

    return;
    }

/*******************************************************************************
*
* forwarding - send DHCP/BOOTP message to target address
*
* This routine relays a DHCP or BOOTP message to the specified target address.
*
* RETURNS: N/A
*
* ERRNO: N/A
*
* NOMANUAL
*/

static STATUS forwarding
    (
    struct server *srvp
    )
    {
    int i, n;
    int sockfd = -1;
    int msgsize = DHCPLEN (dhcpMsgIn.udp);
    struct sockaddr_in tmp_addr, serv_addr;
    int result;
#ifdef DHCPR_DEBUG
    char output [INET_ADDR_LEN];
#endif
    BOOL    bNewFd = FALSE;

    if (dhcpr_socket_fd < 0)
    {
        bNewFd = TRUE;
    }
    else
    {
        sockfd = dhcpr_socket_fd;
    }

    if (bNewFd == TRUE)
    {
    /* This portion of code is orginally from WindRiver. Since we will create
       a task running at the background all the time, we will not dynamically
       create socket.
    */
    if ( (sockfd = socket (AF_INET, SOCK_DGRAM, 0)) < 0) 
        {
#ifdef DHCPR_DEBUG
        logMsg ("socket() error in forwarding()\n", 0, 0, 0, 0, 0, 0);
#endif
        return (ERROR);
        }
    bzero ((char *)&tmp_addr, sizeof (tmp_addr));
    tmp_addr.sin_family = AF_INET;
    tmp_addr.sin_addr.s_addr = htonl (INADDR_ANY);
    /*tmp_addr.sin_port = dhcpc_port;*/  /* Servers expect client port. */
	tmp_addr.sin_port = dhcps_port;
    if (bind (sockfd, (struct sockaddr *) &tmp_addr, sizeof (tmp_addr)) < 0) 
        {
#ifdef DHCPR_DEBUG
        logMsg("bind() error in forwarding()\n", 0, 0, 0, 0, 0, 0);
#endif
        close (sockfd);
        return (ERROR);
        }
    }
    

    dhcprMsgOut.dhcp = dhcpMsgIn.dhcp;

    dhcprFwdMsg.ether = dhcpMsgIn.ether;
    dhcprFwdMsg.ip = dhcpMsgIn.ip;
    dhcprFwdMsg.udp = dhcpMsgIn.udp;
    dhcprFwdMsg.dhcp = dhcpMsgIn.dhcp;

    result = setsockopt (sockfd, SOL_SOCKET, SO_SNDBUF, 
                         (char *)&msgsize, sizeof (msgsize));
    if (result < 0) 
        {
#ifdef DHCPR_DEBUG
        logMsg ("Warning: can't set send buffer in forwarding().\n",
                 0, 0, 0, 0, 0, 0);
#endif
        if (bNewFd == TRUE)
            close (sockfd);

        return (ERROR);
        }
  
    n = sizeof(serv_addr);
    for (i = 0; i < dhcpNumTargets; i++) 
        {
        bzero ((char *)&serv_addr, n);
        serv_addr.sin_family = AF_INET;
        serv_addr.sin_addr.s_addr = srvp->ip.s_addr;
        serv_addr.sin_port = dhcps_port;
#ifdef DHCPR_DEBUG
        inet_ntoa_b (srvp->ip, output);
        printf ("Server relay: sending %d bytes to %s.\n", 
                msgsize, output);
#endif
        sendto (sockfd, (char *)dhcprMsgOut.dhcp, msgsize, 0, 
                  (struct sockaddr *)&serv_addr, n);
        if (Dhcpr_BIPoA_SubnetBroadcast() == FALSE)
        {
            Dhcpr_forwarding(&dhcprFwdMsg, srvp->ip.s_addr);
        }
        srvp = srvp->next;
        }
    if (bNewFd == TRUE)
        close (sockfd);

    if (Dhcpr_BIPoA_SubnetBroadcast() == TRUE)
    {
        Dhcpr_forwarding(&dhcprFwdMsg, 0xffffffff);
    }
    return (OK);
    }

/*******************************************************************************
*
* dhcpClientRelay - send DHCP/BOOTP replies to client
*
* This routine relays a DHCP or BOOTP reply to the client which generated
* the initial request. The routine accesses global pointers already set to 
* indicate the reply. All messages are discarded after the hops field exceeds 
* 16 to comply with the relay agent behavior specified in RFC 1542. The relay 
* agent normally discards such messages before this routine is called. They 
* will only be received by this routine if the user ignores the instructions 
* in the manual and sets the value of DHCP_MAX_HOPS higher than 16.
*
* RETURNS: N/A
*
* ERRNO: N/A
*
* NOMANUAL
*/

void dhcpClientRelay
    (
    struct if_info *list,
    int dhcplen
    )
    {
    int i, total;
    char devName [10];
    struct if_info *ifp;
    int msgtype = 0;
    char *option = NULL;
    struct ifnet *pIf;
    STATUS result;

#ifdef DHCPR_DEBUG
    char srcAddr [INET_ADDR_LEN];
    char dstAddr [INET_ADDR_LEN];
#endif

    if (dhcpMsgIn.dhcp->hops >= 17)    /* RFC limits to 16 hops - ignore. */
        return;

    ifp = list;
    while (ifp != NULL) 
        {
        if (ifp->ipaddr.s_addr == dhcpMsgIn.dhcp->giaddr.s_addr)
            break;
        ifp = ifp->next;
        }

    if (ifp == NULL) 
        { 
#ifdef DHCPR_DEBUG
        inet_ntoa_b (dhcpMsgIn.ip->ip_src, srcAddr);

        logMsg ("Warning: DHCP message from server(%s) has no interface.\n",
	         srcAddr, 0, 0, 0, 0, 0);
#endif
        return;
        }

    sprintf (devName, "%s%d", ifp->name, ifp->unit);

    pIf = ifunit (devName);
    if (pIf == NULL)
        {
#ifdef DHCPR_DEBUG
        logMsg ("Warning: couldn't access network interface %s.\n", 
                 (int)ifp->name, 0, 0, 0, 0, 0);
#endif
        return;
        }


    if ( (option = pickup_opt (dhcpMsgIn.dhcp, dhcplen, _DHCP_MSGTYPE_TAG)) 
          != NULL) 
        msgtype = *OPTBODY(option);

     /* There is already space for ETHER, IP, and UDP headers. */

    dhcprMsgOut.dhcp = dhcpMsgIn.dhcp;
    dhcprMsgOut.udp = dhcpMsgIn.udp;
    dhcprMsgOut.ip = dhcpMsgIn.ip;
    dhcprMsgOut.ether = dhcpMsgIn.ether;

#ifdef DHCPR_DEBUG
    logMsg ("Client relay: Headers found.\n", 0, 0, 0, 0, 0, 0);
#endif

    if (dhcpMsgIn.dhcp->htype != ETHER || dhcpMsgIn.dhcp->hlen != 6)
        return;

#ifdef DHCPR_DEBUG
    logMsg ("Setting parameters.\n", 0, 0, 0, 0, 0, 0);
#endif

    if (ISBRDCST (dhcpMsgIn.dhcp->flags)) 
        {
#ifdef DHCPR_DEBUG
        logMsg ("Broadcasting reply.\n", 0, 0, 0, 0, 0, 0);
#endif
        for (i = 0; i < 6; i++) 
            {
            dhcprMsgOut.ether->ether_dhost[i] = 0xff;
            dhcprMsgOut.ether->ether_shost[i] = ifp->haddr[i];
            }
        dhcprMsgOut.ip->ip_dst.s_addr = 0xffffffff;
        dhcprMsgOut.ip->ip_src.s_addr = ifp->ipaddr.s_addr;
        } 
    else 
        {
        for (i = 0; i < 6; i++) 
            {
            dhcprMsgOut.ether->ether_dhost[i] = dhcpMsgIn.dhcp->chaddr[i];
            dhcprMsgOut.ether->ether_shost[i] = ifp->haddr[i];
            }
        if (msgtype == DHCPNAK)
            dhcprMsgOut.ip->ip_dst.s_addr = 0xffffffff;
        else
            dhcprMsgOut.ip->ip_dst.s_addr = dhcpMsgIn.dhcp->yiaddr.s_addr;
        dhcprMsgOut.ip->ip_src.s_addr = ifp->ipaddr.s_addr;
#ifdef DHCPR_DEBUG
        inet_ntoa_b (dhcprMsgOut.ip->ip_src, srcAddr);
        inet_ntoa_b (dhcprMsgOut.ip->ip_dst, dstAddr);
        logMsg ("Sending reply from %s to %s.\n", srcAddr, dstAddr,
                 0, 0, 0, 0);
        logMsg ("Link layer source: %02x:%02x:%02x:%02x:%02x:%02x.\n",
                 dhcprMsgOut.ether->ether_shost[0],
                 dhcprMsgOut.ether->ether_shost[1],
                 dhcprMsgOut.ether->ether_shost[2],
                 dhcprMsgOut.ether->ether_shost[3],
                 dhcprMsgOut.ether->ether_shost[4],
                 dhcprMsgOut.ether->ether_shost[5]);
        logMsg ("Link layer destination: %02x:%02x:%02x:%02x:%02x:%02x.\n",
                 dhcprMsgOut.ether->ether_dhost[0],
                 dhcprMsgOut.ether->ether_dhost[1],
                 dhcprMsgOut.ether->ether_dhost[2],
                 dhcprMsgOut.ether->ether_dhost[3],
                 dhcprMsgOut.ether->ether_dhost[4],
                 dhcprMsgOut.ether->ether_dhost[5]);
#endif
        }
    dhcprMsgOut.ether->ether_type = ETHERTYPE_IP;
    dhcprMsgOut.udp->uh_sport = dhcps_port;
    dhcprMsgOut.udp->uh_dport = dhcpc_port;
    dhcprMsgOut.udp->uh_ulen = htons (dhcplen + UDPHL);
    dhcprMsgOut.udp->uh_sum = get_udpsum (dhcprMsgOut.ip, dhcprMsgOut.udp);
    dhcprMsgOut.ip->ip_v = IPVERSION;
    dhcprMsgOut.ip->ip_hl = IPHL >> 2;
    dhcprMsgOut.ip->ip_tos = 0;
    dhcprMsgOut.ip->ip_len = htons (dhcplen + UDPHL + IPHL);
    dhcprMsgOut.ip->ip_id = dhcprMsgOut.udp->uh_sum;
    dhcprMsgOut.ip->ip_ttl = 0x20;                       /* XXX */
    dhcprMsgOut.ip->ip_p = IPPROTO_UDP;

    total = dhcplen + IPHL + UDPHL;
    if (total <= ETHERMTU) 
        {
        dhcprMsgOut.ip->ip_off = 0;
        dhcprMsgOut.ip->ip_sum = get_ipsum (dhcprMsgOut.ip);
#ifdef DHCPR_DEBUG
        logMsg ("Writing %d bytes.\n", total, 0, 0, 0, 0, 0);
#endif
        result = etherOutput (pIf, dhcprMsgOut.ether, (char *)dhcprMsgOut.ip,
                              total);
#ifdef DHCPR_DEBUG
       if (result != OK)
            logMsg ("Error %d relaying message to client.\n", errno, 
                    0, 0, 0, 0, 0);
#endif
        } 
    else 
        {
#define  MTU  (ETHERMTU - IPHL)
        char *n, *end, *begin;
        char sendbuf [ETHERMTU];
        struct iovec sbufvec[2];

        sbufvec[0].iov_base = (char *)dhcprMsgOut.ether;
        sbufvec[0].iov_len = ETHERHL;
        begin = (char *) dhcprMsgOut.udp;
        end = &begin [total];
        sbufvec[1].iov_base = sendbuf;

        for (n = begin; n < end; n += MTU) 
            {
            if ((end - n) >= MTU) 
                {
                sbufvec[1].iov_len = MTU;
                dhcprMsgOut.ip->ip_len = htons (ETHERMTU);
                dhcprMsgOut.ip->ip_off = htons(IP_MF | ((((n - begin) / 8)) & 0x1fff));
                dhcprMsgOut.ip->ip_sum = get_ipsum(dhcprMsgOut.ip);
                bcopy ( (char *)dhcprMsgOut.ip, sendbuf, IPHL);
                bcopy ( (char *)n, &sendbuf [IPHL], MTU); 
                } 
            else 
                {
                sbufvec[1].iov_len = end - n;
                dhcprMsgOut.ip->ip_len = htons (IPHL + end - n);
                dhcprMsgOut.ip->ip_off = htons(((n - begin) / 8) & 0x1fff);
                dhcprMsgOut.ip->ip_sum = get_ipsum(dhcprMsgOut.ip);
                bcopy ( (char *)dhcprMsgOut.ip, sendbuf, IPHL);
                bcopy ( (char *)n, &sendbuf [IPHL], end - n); 
                }
            etherOutput (pIf, dhcprMsgOut.ether,
                         sendbuf, dhcprMsgOut.ip->ip_len);
            }
        }
    return;
    }

/*******************************************************************************
*
* read_server_db - read IP addresses to receive relayed packets
*
* This routine extracts the IP address entries hard-coded into usrNetwork.c as 
* dhcpTargetTbl[] and stores them in a circular linked list. Each of these 
* address entries must be on a different subnet from the calling server
* or relay agent. Each entry in the list will receive a copy of every DHCP or
* BOOTP message arriving at the client port.
*
* RETURNS: N/A
*
* ERRNO: N/A
*
* NOMANUAL
*/

void read_server_db (int dbsize)
    {
    char targetIP [sizeof ("255.255.255.255")];
    struct server *srvp = NULL;
    struct server *lastp = NULL;
    int loop;

    /* Read IP addresses of target servers or relay agents. */

    for (loop = 0; loop < dbsize; loop++)
        {
        sprintf (targetIP, "%s", pDhcpRelayTargetTbl [loop].pAddress);

        srvp = (struct server *)calloc (1, sizeof (struct server));
        if (srvp == NULL) 
            {
#ifdef DHCPR_DEBUG
            logMsg ("Memory allocation error reading relay target database.\n", 
                     0, 0, 0, 0, 0, 0);
#endif
            break;
            }
   
        srvp->ip.s_addr = inet_addr (targetIP);
        if (srvp->ip.s_addr == -1) 
            {
#ifdef DHCPR_DEBUG
            logMsg ("Conversion error reading relay target database.\n",
                     0, 0, 0, 0, 0, 0);
#endif
            free (srvp);
            continue;
            }
        dhcpNumTargets++;
        srvp->next = pDhcpTargetList;
        if (pDhcpTargetList == NULL)
            lastp = srvp;
        pDhcpTargetList = srvp;
        }

    if (lastp == NULL)                /* No target DHCP servers. */
        return;

    /* Create circular list of DHCP server addresses. */

    lastp->next = pDhcpTargetList;

    logMsg ("read %d entries from relay target database", dhcpNumTargets, 
             0, 0, 0, 0, 0);
    return;
    }

