/* dhcpc.c - DHCP client runtime finite state machine definition */

/* Copyright 1984 - 1997 Wind River Systems, Inc. */
#include "copyright_wrs.h"

/*
modification history
____________________
01f,26aug97,spm  major overhaul: reorganized code and changed user interface
                 to support multiple leases at runtime
01e,06aug97,spm  removed parameters linked list to reduce memory required
01d,18apr97,spm  added conditional include DHCPC_DEBUG for displayed output
01c,07apr97,spm  modified control function to return ERROR, changed docs
01b,29jan97,spm  removed all functions except state machine definition
01a,03oct96,spm  created by modifying WIDE project DHCP Implementation
*/

/*
DESCRIPTION
This library contains the control function for the DHCP client finite
state machine when executing during runtime.

INCLUDE_FILES: dhcpcLib.h
*/

/*
 * WIDE Project DHCP Implementation
 * Copyright (c) 1995 Akihiro Tominaga
 * Copyright (c) 1995 WIDE Project
 * All rights reserved.
 *
 * Permission to use, copy, modify and distribute this software and its
 * documentation is hereby granted, provided only with the following
 * conditions are satisfied:
 *
 * 1. Both the copyright notice and this permission notice appear in
 *    all copies of the software, derivative works or modified versions,
 *    and any portions thereof, and that both notices appear in
 *    supporting documentation.
 * 2. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by WIDE Project and
 *      its contributors.
 * 3. Neither the name of WIDE Project nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE DEVELOPER ``AS IS'' AND WIDE
 * PROJECT DISCLAIMS ANY LIABILITY OF ANY KIND FOR ANY DAMAGES
 * WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE. ALSO, THERE
 * IS NO WARRANTY IMPLIED OR OTHERWISE, NOR IS SUPPORT PROVIDED.
 *
 * Feedback of the results generated from any improvements or
 * extensions made to this software would be much appreciated.
 * Any such feedback should be sent to:
 *
 *  Akihiro Tominaga
 *  WIDE Project
 *  Keio University, Endo 5322, Kanagawa, Japan
 *  (E-mail: dhcp-dist@wide.ad.jp)
 *
 * WIDE project has the rights to redistribute these changes.
 */

/* includes */

#include "vxWorks.h"
#include "logLib.h"

#include "stdio.h"

#include "stdlib.h"
#include "netinet/in.h"
#include "netinet/udp.h"
#include "netinet/ip.h"
#include "netinet/if_ether.h"

#include "dhcp/dhcpcCommonLib.h"
#include "dhcp/dhcpcInternal.h"
#include "dhcp/dhcpcStateLib.h"

#include "dhcpc/h/dhcpc_api.h"

/*******************************************************************************
*
* dhcp_client_setup - initialize WIDE project data structures for client
*
* This routine initializes the WIDE project data structures used by all leases.
* It assigns functions to handle each of the states in the DHCP client's finite
* state machine, and invokes the initialize() routine to set up the structures
* used for network data transfer. It is called by the dhcpcLibInit() routine
* during the overall initialization of the DHCP client library.
*
* RETURNS: OK if setup completed, or ERROR otherwise.
*
* ERRNO: N/A
*
* NOMANUAL
*/

STATUS dhcp_client_setup
    (
    int 	serverPort, 	/* port used by DHCP servers */
    int 	clientPort 	/* port used by DHCP clients */
    )
    {
    int retval = 0;

    /* Assign routines for processing client's finite state machine. */

    fsm [INIT] = init;
    fsm [WAIT_OFFER] = wait_offer;
    fsm [SELECTING] = selecting;
    fsm [REQUESTING] = requesting;
    fsm [BOUND] = bound;
    fsm [RENEWING] = renewing;
    fsm [REBINDING] = rebinding;
    fsm [INIT_REBOOT] = init_reboot;
    fsm [VERIFY] = verify;
    fsm [REBOOTING] = reboot_verify;
    fsm [VERIFYING] = reboot_verify;

    /* Create data used for message transfer by all leases. */

    if ( (retval = initialize (serverPort, clientPort)) < 0)
        return (ERROR);

    return (OK);
    }

/*******************************************************************************
*
* dhcp_client - Exercise client finite state machine
*
* This routine triggers the finite state machine when a dhcpcBind() call is
* issued from the API before the lease identified by <pCookie> is established.
* It should only be called internally.
* .IP
* The lease negotiation will be performed synchronously or asynchronously,
* depending on the setting of the lease-specific waitFlag parameter. If
* that flag is set, the routine executes synchronously by waiting until the
* lease negotiation completes, then returning the results to the caller.
* This situation always occurs when renewing a lease obtained during system
* boot. The client library always applies any configuration parameters
* associated with that lease to the network interface used for transmission.
* .IP
* This routine initiates the lease negotiation by adding a bind request to
* the event ring which is monitored by the client monitor task. Once the
* negotiation completes, any event hook registered for the lease will be
* invoked by the state machine to allow application-specific processing
* of any parameters obtained.
* interface.
* .IP
* If automatic configuration was requested during the lease initialization,
* the address settings of the transmission network interface will change
* without warning throughout the lifetime of the lease. In particular, the
* initial state of the client's finite state machine will reset that interface
* to use a reserved IP address, preventing access from any other source.
* Notification after the settings are changed will only be available if an
* event hook is installed for the lease.
*
* RETURNS: 0 if processing completed, or value of failed state on error.
*
* ERRNO: N/A
*
* NOMANUAL
*/

int dhcp_client
    (
    void * 	pCookie 	/* lease identifier from dhcpcInit() routine */
    )
    {
    LEASE_DATA * 	pLeaseData;
    BOOL 		syncFlag;
    STATUS 		result;

    /*
     * Use the cookie to access the lease-specific data structures. For now,
     * just typecast the cookie. This translation could be replaced with a more
     * sophisticated lookup at some point.
     */

    pLeaseData = (LEASE_DATA *)pCookie;

    /* Set up variables used on a lease-specific basis. */

         /* Reseed random number generator used for retransmission backoffs. */

    srand (generate_xid (&pLeaseData->ifData));

    if (pLeaseData->autoConfig || pLeaseData->leaseType == DHCP_AUTOMATIC)
        {
        /*
         * Disable network interface until negotiation completes
         * if it used the address information provided by the lease.
         */

        reset_if (&pLeaseData->ifData);
        }

    pLeaseData->prevState = pLeaseData->currState = INIT_REBOOT;

    syncFlag = pLeaseData->waitFlag; /* Assign lease synchronization status. */

    /* Add an event to the message queue to start the lease negotiation. */

    result = dhcpcEventAdd (DHCP_USER_EVENT, DHCP_USER_BIND, pCookie, FALSE);
    if (result == ERROR)
        return (-1);

    /* For synchronous lease negotiation, wait for the completion signal. */

    if (syncFlag)
        semTake (pLeaseData->leaseSem, WAIT_FOREVER);

    return (0);   /* Request successful. */
    }


