/* dhcpr.c - DHCP relay agent library */

/* Copyright 1984 - 1997 Wind River Systems, Inc. */
#include "copyright_wrs.h"

/*
modification history
____________________
01d,06oct97,spm  removed reference to deleted endDriver global; replaced with
                 support for dynamic driver type detection
01c,06may97,spm  changed memory access to align IP header on four byte boundary
01b,28apr97,spm  allowed user to change DHCP_MAX_HOPS setting
01a,07apr97,spm  created by modifying WIDE project DHCP implementation
*/

/*
DESCRIPTION
This library implements the relay agent of the Dynamic Host Configuration 
Protocol. It will transfer all DHCP or BOOTP messages arriving on the
client port of the specified network interfaces across subnet boundaries to
the IP addresses of other DHCP relay agents or DHCP servers.

INCLUDE_FILES: dhcprLib.h
*/

/*
 * WIDE Project DHCP Implementation
 * Copyright (c) 1995 Akihiro Tominaga
 * Copyright (c) 1995 WIDE Project
 * All rights reserved.
 *
 * Permission to use, copy, modify and distribute this software and its
 * documentation is hereby granted, provided only with the following
 * conditions are satisfied:
 *
 * 1. Both the copyright notice and this permission notice appear in
 *    all copies of the software, derivative works or modified versions,
 *    and any portions thereof, and that both notices appear in
 *    supporting documentation.
 * 2. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by WIDE Project and
 *      its contributors.
 * 3. Neither the name of WIDE Project nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE DEVELOPER ``AS IS'' AND WIDE
 * PROJECT DISCLAIMS ANY LIABILITY OF ANY KIND FOR ANY DAMAGES
 * WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE. ALSO, THERE
 * IS NO WARRANTY IMPLIED OR OTHERWISE, NOR IS SUPPORT PROVIDED.
 *
 * Feedback of the results generated from any improvements or
 * extensions made to this software would be much appreciated.
 * Any such feedback should be sent to:
 * 
 *  Akihiro Tominaga
 *  WIDE Project
 *  Keio University, Endo 5322, Kanagawa, Japan
 *  (E-mail: dhcp-dist@wide.ad.jp)
 *
 * WIDE project has the rights to redistribute these changes.
 */

/* includes */

#include <stdio.h>
#include <stdlib.h>
#include <sys/ioctl.h>

#include <netinet/in.h>
#include <arpa/inet.h>

#include "etherLib.h"
#include "logLib.h"
#include "sockLib.h"
#include "ioLib.h"
#include "muxLib.h"

#include "dhcprLib.h"
#include "dhcp/dhcp.h"
#include "dhcp/common.h"
#include "dhcp/common_subr.h"

/* globals */

IMPORT BOOL dhcprInputHook (struct ifnet*, char*, int);
IMPORT void Dhcpr_MsgSemTake(void);
IMPORT void Dhcpr_MsgSemGive(void);

struct msg dhcprMsgIn;
IMPORT struct if_info *dhcprIntfaceList;
IMPORT int dhcpMaxHops;

static char *rbufp;                   /* receive buffer */
LOCAL unsigned char dhcpCookie[] = RFC1048_MAGIC;

/* forward declarations */

void dhcpServerRelay (struct if_info *);
void dhcpClientRelay (struct if_info *, int);

/*******************************************************************************
*
* dhcprStart - monitor specified network interfaces
*
* This routine monitors the interfaces specified by the user for incoming
* DHCP or BOOTP messages. It is the entry point for the relay agent task and 
* should only be called internally.
*
* RETURNS: N/A
*
* ERRNO: N/A
*
* NOMANUAL
*/

void dhcprStart (void)
    {
    struct if_info *ifp = NULL;          /* pointer to interface */
    int n = 0;
    BOOL bsdDrvFlag = TRUE;

    /* 
     * Begin receiving appropriate Ethernet frames from every interface. 
     * Currently, the same input hook routine must be added once for all BSD
     * driver, if any, and once for each END-style driver. Eventually, DHCP 
     * will be registered as a protocol and the input hooks will be
     * removed.
     */

    ifp = dhcprIntfaceList;

    while (ifp != NULL)
        {
        if (muxDevExists (ifp->name, ifp->unit))
            etherInputHookAdd (dhcprInputHook, ifp->name, ifp->unit);
        else if (bsdDrvFlag)
            {
            /* Add the input hook once for all BSD 4.4 drivers. */

            etherInputHookAdd (dhcprInputHook, NULL, ifp->unit);
            bsdDrvFlag = FALSE;
            }
        ifp = ifp->next;
        }

  /****************************
   * Main loop                *
   * Process incoming message *
   ****************************/
  
    FOREVER
        {
        /* select and read from interfaces */

        if ( (ifp = read_interfaces (&dhcprIntfaceList, &n)) == NULL) 
            continue;

        /*
         * Access at offset to provide 4-byte alignment for IP header
         * needed by Sun BSP's.
         */

        rbufp = &ifp->buf [DHCPS_OFF];

        /* convert buffer to struct ether, ip, udp and dhcp */
        dhcprMsgIn.ether = (struct ether_header *)rbufp;
        dhcprMsgIn.ip = (struct ip *)&rbufp [ETHERHL];
#if 0   
        /* This portion code is the orignal from VxWorks */
        if ( (ntohs (dhcprMsgIn.ip->ip_off) & 0x1fff) == 0 &&
              ntohs(dhcprMsgIn.ip->ip_len) >= DFLTBOOTPLEN + UDPHL + IPHL &&
              (dhcprMsgIn.ip->ip_dst.s_addr == 0xffffffff ||
               dhcprMsgIn.ip->ip_dst.s_addr == ifp->ipaddr.s_addr) &&
               check_ipsum(dhcprMsgIn.ip))
            dhcprMsgIn.udp = (struct udphdr *)&rbufp [ETHERHL +
                                                      dhcprMsgIn.ip->ip_hl *
                                                      _LEN];
#else
       if ( (ntohs (dhcprMsgIn.ip->ip_off) & 0x1fff) == 0 &&
              ntohs(dhcprMsgIn.ip->ip_len) >= DFLTBOOTPLEN + UDPHL + IPHL)
       {
            struct if_info *if_info_p = NULL;
            BOOL   bMatchIpAddr = FALSE;

            if ( dhcprMsgIn.ip->ip_dst.s_addr != 0xffffffff )
            {
                for (if_info_p = dhcprIntfaceList;
                        if_info_p != NULL && bMatchIpAddr == FALSE;
                        if_info_p = if_info_p->next)
                    if (dhcprMsgIn.ip->ip_dst.s_addr == if_info_p->ipaddr.s_addr)
                        bMatchIpAddr = TRUE;
            }
            else
                bMatchIpAddr = TRUE;

            if ( (bMatchIpAddr == TRUE) && check_ipsum(dhcprMsgIn.ip))
                dhcprMsgIn.udp = (struct udphdr *)&rbufp [ETHERHL +
                                                      dhcprMsgIn.ip->ip_hl *
                                                      WORD_LEN];
       }
#endif
        else
            continue;

        if (ntohs (dhcprMsgIn.udp->uh_ulen) >= DFLTBOOTPLEN + UDPHL &&
                   dhcprMsgIn.udp->uh_dport == dhcps_port &&
                   check_udpsum (dhcprMsgIn.ip, dhcprMsgIn.udp))
            dhcprMsgIn.dhcp = (struct dhcp *) &rbufp [ETHERHL +
                                               dhcprMsgIn.ip->ip_hl * WORD_LEN +
                                               UDPHL];
        else
            continue;

        /* final packet check */

        if (dhcprMsgIn.dhcp->hops <= dhcpMaxHops &&
              bcmp (dhcprMsgIn.dhcp->options, 
                    (char *)dhcpCookie, MAGIC_LEN) == 0)
            {
            Dhcpr_MsgSemTake();
            dhcpMsgIn.ether = dhcprMsgIn.ether;
            dhcpMsgIn.ip = dhcprMsgIn.ip;
            dhcpMsgIn.udp = dhcprMsgIn.udp;
            dhcpMsgIn.dhcp = dhcprMsgIn.dhcp;
            if (dhcpMsgIn.dhcp->op == BOOTREQUEST)
                dhcpServerRelay (ifp);     /* process the packet */
            else if (dhcpMsgIn.dhcp->op == BOOTREPLY)
                dhcpClientRelay (dhcprIntfaceList, DHCPLEN (dhcprMsgIn.udp));
            Dhcpr_MsgSemGive();
            }
        }
    }
