/* dhcpcLib.c - Dynamic Host Configuration Protocol (DHCP) run-time client API */

/* Copyright 1984 - 1997 Wind River Systems, Inc.  */
#include "copyright_wrs.h"

/*
modification history
____________________
01s,10dec97,kbw  fixed minor spelling issues in man pages
01r,04dec97,spm  added code review modifications; increased stack size for
                 monitor task to 50% over maximum for ss5 board
01q,27oct97,spm  corrected name field to contain device name only
01p,21oct97,kbw  fixed minor spelling issues in man pages
01o,06oct97,spm  removed reference to deleted endDriver global; replaced with
                 support for dynamic driver type detection
01n,25sep97,gnn  SENS beta feedback fixes
01m,02sep97,spm  removed unused global variable for event hook routine
01l,26aug97,spm  major overhaul: reorganized code and changed user interface
                 to support multiple leases at runtime
01k,12aug97,gnn  changes necessitated by MUX/END update.
01j,06aug97,spm  removed parameters linked list to reduce memory required;
                 corrected minor error in man pages introduced by 01i revision
01i,30jul97,kbw  fixed man page problems found in beta review
01h,15jul97,spm  fixed byte ordering of netmask from DHCP client (SPR #8739)
01g,10jun97,spm  fixed code layout and error in client lease indicator
01f,02jun97,spm  changed DHCP option tags to prevent name conflicts (SPR #8667)
                 and updated man pages
01e,30apr97,spm  changed dhcpcOptionGet() to return length; expanded man pages
01d,15apr97,kbw  fixed format problems in man page text, changed some wording
01c,07apr97,spm  altered to use current value of configAll.h defines, fixed
                 user requests, inserted code previously in usrNetwork.c,
                 rewrote documentation
01b,29jan97,spm  added END driver support and modified to fit coding standards.
01a,03oct96,spm  created by modifying WIDE Project DHCP Implementation.
*/

/*
DESCRIPTION
This library implements the run-time access to the client side of the
Dynamic Host Configuration Protocol (DHCP).  DHCP is an extension of BOOTP.
Like BOOTP, the protocol allows a host to initialize automatically by obtaining
its IP address, boot file name, and boot host's IP address over a network.
Additionally, DHCP provides a client with the complete set of parameters
defined in the Host Requirements RFCs and allows automatic reuse of network
addresses by specifying individual leases for each set of configuration
parameters.  The compatible message format allows DHCP participants to interact
with BOOTP participants.  The dhcpcLibInit() routine links this library into
the VxWorks image. This happens automatically if INCLUDE_DHCPC is defined at
the time the image is built.

CONFIGURATION INTERFACE
When used during run time, the DHCP client library establishes and maintains
one or more DHCP leases.  Each lease provides access to a set of configuration
parameters.  If requested, the parameters retrieved will be used to reconfigure
the associated network interface, but may also be handled separately through
an event hook.  The dhcpcEventHookAdd() routine specifies a function which is
invoked whenever the lease status changes.  The dhcpcEventHookDelete() routine
will disable that notification.  The automatic reconfiguration must be limited
to one lease for a particular network interface.  Otherwise, multiple leases
would attempt to reconfigure the same device, with unpredictable results.

HIGH-LEVEL INTERFACE
To access the DHCP client during run time, an application must first call
the dhcpcInit() routine with a pointer to the network interface to be used
for communication with a DHCP server.  Each call to the initialization
routine returns a unique identifier to be used in subsequent calls to the
DHCP client routines.  Next, the application must specify a client identifier
for the lease using the dhcpcOptionSet() call.  Typically, the link-level
hardware address is used for this purpose.  Additional calls to the option set
routine may be used to request specific DHCP options.  After all calls to that
routine are completed, a call to dhcpcBind() will retrieve a set of
configuration parameters according to the client-server interaction detailed
in RFC 1541.

Each sequence of the three function calls described above, if successful,
will retrieve a set of configuration parameters from a DHCP server.  The
dhcpcServerGet() routine retrieves the address of the server that provided a
particular lease.  The dhcpcTimerGet() routine will retrieve the current values
for both lease timers.

Alternatively, the dhcpcParamsGet() and dhcpcOptionGet() routines will access
any options provided by a DHCP server.  In addition to the lease identifier
obtained from the initialization routine, the dhcpcParamsGet() routine accepts
a parameter descriptor structure that selects any combination of the options
described in RFC 1533 for retrieval.  Similarly, the dhcpcOptionGet() routine
retrieves the values associated with a single option.

LOW-LEVEL INTERFACE
This library also contains several routines which explicitly generate DHCP
messages.  The dhcpcVerify() routine causes the client to renew a particular
lease, regardless of the time remaining.  The dhcpcRelease() routine
relinquishes the specified lease.  The associated parameters are no longer
valid.  If those parameters were used by the underlying network device, the
routine also shuts off all network processing for that interface.  Finally,
the dhcpcShutdown() routine will release all active leases and disable all
the DHCP client library routines.

OPTIONAL INTERFACE
The dhcpcCacheHookAdd() routine registers a function that the client will
use to store and retrieve lease data.  The client can then re-use this
information if it is rebooted.  The dhcpcCacheHookDelete() routine prevents
the re-use of lease data.  Initially, a function to access permanent storage
is not provided.

INTERNAL
The diagram below defines the structure chart of dhcpcLib.

     |            |              |
     v            v              v

 dhcpcLibInit  dhcpcSetup    dhcpcConfigSet -------\

     |                       /    |    \            \

     v                      |     |     |            |

  dhcpcMon                  v     |     v            |
(spawned task)
                        dhcpcInit | dhcpcBind        |

                                  |                  |

                                  |                  |

                                  |                  |

                                  v                  v

                            dhcpcOptionSet     dhcpcParamsGet

This library provides a wrapper for the WIDE project DHCP code found in the
directory /vobs/wpwr/target/src/dhcp, which contains the state machine and
other supporting functions.  The monitor task redirects incoming messages,
timeout events, and user requests to the appropriate state in the state
machine.  The input hook used to retrieve incoming messages and the
monitor task routines are both in the dhcpcCommonLib module in this directory.

The current RFC specification does not allow users to obtain parameters
with DHCP without also obtaining a new IP address.  This limitation requires
network shutdown if the lease used by an interface is not maintained.  It also
artificially limits the use of this library to routines involving an active
lease.  The draft RFC for the successor to RFC 1541 adds a new message which
avoids this problem.  Once published, DHCP can safely be used within an
application without risking network shutdown for any interface which was
statically configured during system boot.  The code which shuts down the
network should then be limited to execute conditionally on interfaces which
have no backup source of addressing information.

INCLUDE FILES: dhcpcLib.h

SEE ALSO: RFC 1541, RFC 1533
*/

/* includes */

#include "dhcp/copyright_dhcp.h"
#include "vxWorks.h"
#include "bootLib.h"
#include "sysLib.h"
#include "wdLib.h"
#include "semLib.h"
#include "etherLib.h"
#include "inetLib.h"
#include "rngLib.h"
#include "taskLib.h"
#include "logLib.h"
#include "muxLib.h"

#include "stdio.h"
#include "stdlib.h"
#include "unistd.h"
#include "string.h"
#include "errno.h"
#include "signal.h"
#include "fcntl.h"
#include "sys/types.h"
#include "sys/socket.h"
#include "sys/ioctl.h"
#include "net/if.h"
#include "netinet/in.h"
#include "netinet/in_systm.h"
#include "netinet/if_ether.h"
#include "netinet/ip.h"
#include "netinet/udp.h"
#include "arpa/inet.h"
#include "time.h"

#include "dhcpcLib.h"
#include "dhcp/dhcpcStateLib.h"
#include "dhcp/dhcpcCommonLib.h"

#include "end.h"
#include "ipProto.h"
#include "dhcpc/h/dhcpc_cfg.h"

#define _DHCP_MAX_OPTLEN 	255 	/* Max. number of bytes in an option */

/* externals */

IMPORT RING_ID 	dhcpcEventRing;	/* Ring buffer of DHCP events */
IMPORT RING_ID 	dhcpcMsgRing; 	/* Ring buffer of DHCP messages */
IMPORT SEM_ID 	dhcpcEventSem; 	/* DHCP event notification */
IMPORT SEM_ID 	dhcpcMutexSem; 	/* Protects the DHCP status indicator */
IMPORT int dhcpcTaskId;		/* Identifier for client monitor task */

IMPORT FUNCPTR dhcpcCacheHookRtn;	/* Data storage/retrieval function */
IMPORT SEM_ID  dhcpc_SemId;

/* globals */

LEASE_DATA ** 	dhcpcLeaseList; 	/* List of available cookies. */
int 	dhcpcMaxLeases; 	/* configAll.h #define value */
int 	dhcpcMinLease; 		/* configAll.h #define value */
void * 	pDhcpcBootCookie [DHCPC_MAX_IF_NUM]; 	/* Access to boot-time lease. */
BOOL dhcpcInitialized = FALSE;  /* Client initialized? */
DHCP_LEASE_DATA 	dhcpcBootLease; 	/* Boot-time lease settings. */

int _dhcpcTaskPriority  = 100;      /* Priority level of lease monitor */
int _dhcpcTaskOptions   = 0;       /* Option settings of lease monitor */
int _dhcpcTaskStackSize = 5000;    /* Stack size of lease monitor */

LOCAL int dhcpcOfferTimeout;           /* configAll.h #define value */
LOCAL int dhcpcDefaultLease;           /* configAll.h #define value */

/* forward declarations */

IMPORT void dhcpcMon(void);				/* Monitor lease */
STATUS dhcpcVerify (void *);				/* Renew lease */
IMPORT char *dhcpcOptionFind (LEASE_DATA *, int, int *);   /* Locate option */

/*******************************************************************************
*
* dhcpcLibInit - DHCP client library initialization
*
* This routine creates and initializes the global data structures used by
* the DHCP client library to maintain multiple leases, up to the limit
* specified by the <maxLeases> parameter.  Every subsequent lease attempt will
* collect additional DHCP offers until the interval specified by <offerTimeout>
* expires and will request the lease duration indicated by <defaultLease>.
* This routine must be called before calling any other library routines.  The
* routine is called automatically if INCLUDE_DHCPC is defined at the time the
* system is built and assigns the global lease settings to the values specified
* by DHCPC_SPORT, DHCPC_CPORT, DHCPC_MAX_LEASES, DHCPC_DEFAULT_LEASE,
* and DHCPC_OFFER_TIMEOUT.
*
* RETURNS: OK, or ERROR if initialization fails.
*
* ERRNO:
*  S_dhcpcLib_MEM_ERROR
*
*/

STATUS dhcpcLibInit
    (
    int 	serverPort, 	/* port used by DHCP servers (default 67) */
    int 	clientPort, 	/* port used by DHCP clients (default 68) */
    int     maxLeases, 	/* max number of simultaneous leases allowed */
    int 	offerTimeout, 	/* interval to get additional DHCP offers */
    int 	defaultLease, 	/* default value for requested lease length */
    int 	minLease 	/* minimum accepted lease length */
    )
    {
    int loop, i;
    STATUS result, retval;

    retval = OK;
    if (dhcpcLeaseList != NULL && dhcpcInitialized == TRUE)
    {
        goto out;
    }

    for(i = 0; i < DHCPC_MAX_IF_NUM; i++)
        pDhcpcBootCookie[i] = NULL;

    /* Create and initialize storage for active leases. */
    dhcpcLeaseList = malloc (maxLeases * sizeof (LEASE_DATA *));
    if (dhcpcLeaseList == NULL)
    {
        errno = S_dhcpcLib_MEM_ERROR;
        retval = ERROR;
        goto out;
    }

    for (loop = 0; loop < maxLeases; loop++)
        dhcpcLeaseList [loop] = NULL;

    dhcpcMaxLeases = maxLeases;
    dhcpcDefaultLease = defaultLease;
    dhcpcOfferTimeout = offerTimeout;
    dhcpcMinLease = minLease;

    /* Set up state machine and data structures for input hooks.  */

    result = dhcp_client_setup (serverPort, clientPort);
    if (result == ERROR)
        {
        free (dhcpcLeaseList);
        errno = S_dhcpcLib_MEM_ERROR;
        retval = ERROR;
        goto out;
        }

    /*
     * Create protection semaphore for all lease status indicators.  User
     * requests which require an active lease check this indicator before
     * executing.  A single semaphore is used to reduce system resource
     * requirements, even though this behavior may allow user requests to
     * slightly delay status updates for unrelated leases.
     */

    dhcpcMutexSem = semBCreate (SEM_Q_FIFO, SEM_FULL);
    if (dhcpcMutexSem == NULL)
        {
        free (dhcpcLeaseList);
        errno = S_dhcpcLib_MEM_ERROR;
        retval = ERROR;
        goto out;
        }

    /* Create signalling semaphore for event notification ring.  */

    dhcpcEventSem = semCCreate (SEM_Q_FIFO, 0);
    if (dhcpcEventSem == NULL)
        {
        free (dhcpcLeaseList);
        semDelete (dhcpcMutexSem);
        errno = S_dhcpcLib_MEM_ERROR;
        retval = ERROR;
        goto out;
        }

    /* Create event storage.  */

    dhcpcEventRing = rngCreate (EVENT_RING_SIZE);
    if (dhcpcEventRing == NULL)
        {
        free (dhcpcLeaseList);
        semDelete (dhcpcMutexSem);
        semDelete (dhcpcEventSem);
        errno = S_dhcpcLib_MEM_ERROR;
        retval = ERROR;
        goto out;
        }

    /* Create message storage.  */

    dhcpcMsgRing = rngCreate (MESSAGE_RING_SIZE);
    if (dhcpcMsgRing == NULL)
        {
        free (dhcpcLeaseList);
        semDelete (dhcpcMutexSem);
        semDelete (dhcpcEventSem);
        rngDelete (dhcpcEventRing);
        errno = S_dhcpcLib_MEM_ERROR;
        retval = ERROR;
        goto out;
        }

    dhcpcCacheHookRtn = NULL;

    /*
     * Spawn the monitor task.  The entry routine will process all incoming
     * DHCP messages and timer related or user generated events.
     */

    result = taskSpawn ("tDhcpcTask", _dhcpcTaskPriority, _dhcpcTaskOptions,
                            _dhcpcTaskStackSize, (FUNCPTR) dhcpcMon,
                            0, 0, 0, 0, 0, 0, 0, 0, 0, 0);
    if (result == ERROR)
        {
        free (dhcpcLeaseList);
        semDelete (dhcpcMutexSem);
        semDelete (dhcpcEventSem);
        rngDelete (dhcpcEventRing);
        rngDelete (dhcpcMsgRing);
        errno = S_dhcpcLib_MEM_ERROR;
        retval = ERROR;
        goto out;
        }

    dhcpcTaskId = result;

    dhcpcInitialized = TRUE;

    return retval;

out:
    semGive(dhcpc_SemId);
    return retval;
    }

/*******************************************************************************
*
* dhcpcLeaseGet - read lease information from boot line
*
* If an automatic configuration protocol was used by the bootstrap loader
* (i.e. - SYSFLG_AUTOCONFIG was set), this routine is called to handle
* possible DHCP values in the address string.  If the target address indicated
* by <pAddrString> contains both lease duration and lease origin values, it was
* assigned by a DHCP server and the routine sets the status flag accessed by
* <pDhcpStatus> so that the lease will be verified or renewed.  This routine is
* called when initializing the network during system startup and will not
* function correctly if used in any other context.  Any lease information
* attached to the address string is removed when this routine is executed.
*
* RETURNS: OK if setup completes, or ERROR if the address string is invalid.
*
* ERRNO: N/A
*
* NOMANUAL
*/

STATUS dhcpcLeaseGet
    (
    char * 	pAddrString, 	/* client address string from bootline */
    BOOL * 	pDhcpStatus 	/* DHCP lease values found? */
    )
    {
    char * 	pDelim;
    char * 	pOffset;
    int 	result;

    dhcpcBootLease.lease_duration = 0; 	/* Defaults to expired.  */
    dhcpcBootLease.lease_origin = 0; 	/* Defaults to invalid value.  */

    /* Read DHCP lease values and remove from address string.  */
    result = bootLeaseExtract (pAddrString, &dhcpcBootLease.lease_duration,
                               &dhcpcBootLease.lease_origin);
    if (result == 2)
        *pDhcpStatus = TRUE;    /* DHCP lease values read successfully.  */
    else
        {
        *pDhcpStatus = FALSE;    /* Valid DHCP lease values not present.  */

        if (result == 0)
            {
            /*
             * The ":" field separator for either the netmask or duration was
             * not found, so no DHCP lease values are present.
             */

            return (OK);
            }

        if (result == 1)
            {
            /*
             * Only the lease duration field was present.
             * The DHCP lease values have been removed.
             */

           return (ERROR);
           }

        /*
         * Parsing one of the lease values failed.  Remove
         * any DHCP lease data from the address string.
         */

        /* Find delimiter for netmask.  */

        pOffset = index (pAddrString, ':');

        /*
         * Start at the delimiter for DHCP lease values.  The netmask separator
         * is actually always present at this point, but check for it anyway.
         */

        if (pOffset != NULL)
            {
            pDelim = pOffset + 1;

            /*
             * Find the lease duration tag.  This field separator is also
             * guaranteed to be present, or the extract routine would have
             * returned 0.
             */

            pOffset = index (pDelim, ':');

            /* Remove the DHCP lease values from string.  */

            if (pOffset != NULL)
                *pOffset = EOS;
            }
        return (ERROR);
        }
    return (OK);
    }

/*******************************************************************************
*
* dhcpcConfigSet - set system configuration according to active DHCP lease
*
* This routine verifies or renews any DHCP lease established during the system
* boot.  If a DHCP lease was established, the dhcpcInit() call in this routine
* will setup the necessary data structures and create a cookie identifying the
* boot lease.  The cookie is stored in the pDhcpcBootCookie global variable to
* provide users with access to the boot-time lease.  This routine is called when
* initializing the network from usrNetInit() in usrNetwork.c and will not
* function correctly if executed in any other context.
*
* RETURNS: OK if configuration completed, or ERROR otherwise.
*
* ERRNO: N/A
*
* NOMANUAL
*/
STATUS dhcpcConfigSet
    (
    BOOT_PARAMS *pParams, 	/* structure for parsed bootline */
    int * 		pNetmask, 	/* specified subnet mask */
    int 		dhcpStatus 	/* DHCP or BOOTP used? */
    )
    {
    char 				netDev [BOOT_DEV_LEN + 1], if_name [BOOT_DEV_LEN + 1] ;
    struct dhcp_param 	bootParams;
    BOOL 				backplaneBoot;
    u_long 				result;
    struct ifnet 		*pIf;
    char 				tempString[30];
    LEASE_DATA 			*pLeaseData;
	void				*Cookie;
	int 				i;

    unsigned char  		*ifmac = NULL;
	END_OBJ				*end_obj;
	char 				client_id[10];
	short				num_wan;

    backplaneBoot = FALSE;

    if ( (strncmp (pParams->bootDev, "bp", 2) == 0) ||
        (strncmp (pParams->bootDev, "sm", 2) == 0))
        backplaneBoot = TRUE;

    /* Check for serial interfaces - DHCP requires broadcast-capable devices.  */

    if ( (strncmp (pParams->bootDev, "ppp", 3) == 0) ||
        (strncmp (pParams->bootDev, "sl", 2) == 0))
         {
         if (dhcpStatus == FALSE)    /* No DHCP lease is in use.  */
             return (OK);
         else
             return (ERROR);         /* Can't verify current lease.  */
         }

    if (dhcpStatus == FALSE)
        {
        /*
         * BOOTP reply was selected during system startup or the IP address
         * was assigned manually.  Lease renewal is not needed.
         */
        return (OK);
        }

    /* Verify DHCP lease obtained during boot.  */

    bzero ( (char *)&bootParams, sizeof (struct dhcp_param));

    sprintf (netDev, "%s%d", pParams->bootDev, pParams->unitNum);

    /* Fill in client address obtained from bootline.  */

    if (backplaneBoot)
        result = inet_addr (pParams->bad);
    else
        result = inet_addr (pParams->ead);
    if (result == ERROR)
        {
        if (backplaneBoot)
            printf("Invalid target address \"%s\"\n", pParams->bad);
        else
            printf("Invalid target address \"%s\"\n", pParams->ead);
        return (ERROR);
        }
    dhcpcBootLease.yiaddr.s_addr = result;

    //
    //  Find the cookie
    //
	SYSTEM_NumOfWanGet(&num_wan);
    for (i=0; i < num_wan; i++)
    {
        if (pDhcpcBootCookie[i] != NULL)
        {
            pLeaseData = (LEASE_DATA *)pDhcpcBootCookie[i];
            sprintf (if_name, "%s%d", pLeaseData->ifData.name,
                        pLeaseData->ifData.unit);
            if ( strcmp(if_name, netDev) == 0)
            {
                Cookie = pDhcpcBootCookie[i];
                break;
            }
        }
    }


    if (Cookie == NULL)
        {
        printf ("Error initializing DHCP boot lease.\n");
        return (ERROR);
        }

    /*
     * Use the cookie to access the lease-specific data structures.  For now,
     * just typecast the cookie.  This translation could be replaced with a more
     * sophisticated lookup at some point.
     */
//  pLeaseData = (LEASE_DATA *)pDhcpcBootCookie;
    pLeaseData = (LEASE_DATA *)Cookie;

    /* Set the lease descriptor to identify the boot lease.  */

    pLeaseData->leaseType = DHCP_MANUAL;
    //pLeaseData->leaseType = DHCP_AUTOMATIC;

    // added for avoid the lease time request
    dhcpcOptionSet (Cookie, _DHCP_LEASE_TIME_TAG, 0, 0, NULL);

    // added for subnet mask
    dhcpcOptionSet (Cookie, _DHCP_SUBNET_MASK_TAG, 0, 0, NULL);

    dhcpcOptionSet (Cookie, _DHCP_ROUTER_TAG, 0, 0, NULL);
    dhcpcOptionSet (Cookie, _DHCP_DNS_SERVER_TAG, 0, 0, NULL);

    // added for domain name
    dhcpcOptionSet (Cookie, _DHCP_DNS_DOMAIN_TAG, 0, 0, NULL);

    // added for the hostName of target
    if (pParams->hostName[0] != EOS)
        dhcpcOptionSet (Cookie, _DHCP_HOSTNAME_TAG, 0,
                    sizeof(pParams->hostName), pParams->hostName);

    end_obj = endFindByName(pParams->bootDev, pParams->unitNum);
    ifmac = end_obj->mib2Tbl.ifPhysAddress.phyAddress;
    memcpy(&client_id[0], ifmac, 6);
    dhcpcOptionSet (Cookie, _DHCP_CLIENT_ID_TAG, 1, 6, client_id);

    /* Execute the bind call synchronously to verify the boot lease.  */
//  if (dhcpcBind (Cookie, TRUE) != OK)
    if (dhcpcBind (Cookie, FALSE) != OK)
        {
        printf ("Can't renew DHCP boot lease.\n");
        Cookie = NULL;
        return (ERROR);
        }

    return (OK);
    }

/*******************************************************************************
*
* dhcpcInit - assign network interface and setup lease request
*
* This routine creates the data structures used to obtain a set of parameters
* with DHCP and must be called before each attempt at establishing a DHCP
* lease, but after the dhcpcLibInit() routine has initialized the global data
* structures.  The <pIf> argument indicates the network device which will be
* used for transmission and reception of DHCP messages during the lifetime of
* the lease.  If the <autoConfig> parameter is set to TRUE, any address
* information obtained will automatically be applied to that interface.  The
* specified interface must access a device capable of sending broadcast
* messages.  Currently, only Ethernet devices and the shared-memory network
* drivers are supported.
*
* The routine also uses the <autoConfig> parameter to select the default option
* request list for a lease.  If set to FALSE, no specific lease options are
* requested since any configuration parameters obtained are not intended for
* the underlying network device.  In that case, any specific options required
* may be added to the request list at any time before the corresponding
* dhcpcBind() call.  If <autoConfig> is TRUE, this routine sets the
* configuration parameters to request the minimal address information
* (subnet mask and broadcast address) necessary for reconfiguring the network
* device specified by <pIf>.
*
* The internal lease identifier returned by this routine must be used in
* subsequent calls to the DHCP client library.
*
* NOTE
* This routine is called automatically during system startup if the DHCP
* client was used to obtain the VxWorks boot parameters.  The resulting
* lease will always reconfigure the network boot device.  Therefore, any
* further calls to this routine which specify the network boot device for
* use in obtaining additional DHCP leases must set <autoConfig> to FALSE.
* Otherwise, that device will be unable to maintain a stable configuration.
* The global variable pDhcpcBootCookie provides access to the configuration
* parameters for any DHCP lease created during system startup.
*
* RETURNS: Lease handle for later use, or NULL if lease setup fails.
*
* ERRNO:
*  S_dhcpcLib_NOT_INITIALIZED
*  S_dhcpcLib_NO_DEVICE
*  S_dhcpcLib_BAD_OPTION
*  S_dhcpcLib_MAX_LEASES_REACHED
*  S_dhcpcLib_MEM_ERROR
*
* SEE ALSO
* dhcpcOptionSet(), dhcpcEventHookAdd()
*/

void * dhcpcInit
    (
    struct ifnet *	pIf, 		/* network device used by client */
    BOOL 		autoConfig 	/* reconfigure network device? */
    )
    {
    void * 			pCookie;
    LEASE_DATA *	 	pLeaseData;
    struct dhcp_reqspec * 	pReqSpec;
    int offset;

    if (!dhcpcInitialized)
        {
        errno = S_dhcpcLib_NOT_INITIALIZED;
        return (NULL);
        }

    if (pIf == NULL)
        {
        errno = S_dhcpcLib_NO_DEVICE;
        return (NULL);
        }

    if (autoConfig != TRUE && autoConfig != FALSE)
        {
        errno = S_dhcpcLib_BAD_OPTION;
        return (NULL);
        }

    /* Find an unused entry in the array of lease-specific variables.  */
    for (offset = 0; offset < dhcpcMaxLeases; offset++)
        if (dhcpcLeaseList [offset] == NULL)
            break;

    if (offset == dhcpcMaxLeases)
        {
        errno = S_dhcpcLib_MAX_LEASES_REACHED;
        return (NULL);
        }

    /* Allocate all lease-specific variables.  */

    pLeaseData = (LEASE_DATA *)calloc (1, sizeof (LEASE_DATA));
    if (pLeaseData == NULL)
        {
        errno = S_dhcpcLib_MEM_ERROR;
        return (NULL);
        }

    pLeaseData->initFlag = FALSE;

    pLeaseData->autoConfig = autoConfig;

    /*
     * Use the current data storage hook routine (if any)
     * throughout the lifetime of this lease.
     */

    pLeaseData->cacheHookRtn = dhcpcCacheHookRtn;

    /*
     * For now, use the lease data pointer as the unique lease identifier.
     * This could be changed later to shield the internal structures from
     * the user, but it allows fast data retrieval.
     */

    pCookie = (void *)pLeaseData;

    pReqSpec = &pLeaseData->leaseReqSpec;

    bzero ( (char *)pReqSpec, sizeof (struct dhcp_reqspec));
    bzero ( (char *)&pLeaseData->ifData, sizeof (struct if_info));

    /* Initialize WIDE project global containing network device data.  */

    sprintf (pLeaseData->ifData.name, "%s", pIf->if_name);
    pLeaseData->ifData.unit = pIf->if_unit;
    pLeaseData->ifData.iface = pIf;
    pLeaseData->ifData.endDrvFlag = muxDevExists (pIf->if_name, pIf->if_unit);

    /* Change htype and hlen setup from hard-coded Ethernet to ioctls.  */

    pLeaseData->ifData.haddr.htype = ETHER;
    pLeaseData->ifData.haddr.hlen = _ETHERADDRLEN;

    bcopy ( (char *) ( (struct arpcom *)pIf)->ac_enaddr,
           (char *)&pLeaseData->ifData.haddr.haddr,
           pLeaseData->ifData.haddr.hlen);

    /* set duration of client's wait for multiple offers */

    pReqSpec->waitsecs = dhcpcOfferTimeout;

    /* set default lease - may be overridden with dhcpcOptionSet().  */

    pReqSpec->lease = dhcpcDefaultLease;

    pReqSpec->reqlist.len = 0;    /* No options requested yet.  */

    /*
     * If executing with startup lease, or if automatic configuration is
     * requested, initialize request list with tags for options required
     * by all network interfaces, using RFC 1533 tag values.
     */

    if (pLeaseData->autoConfig || pLeaseData->leaseType == DHCP_AUTOMATIC)
        {
        pReqSpec->reqlist.list [pReqSpec->reqlist.len++] =
                                                        _DHCP_SUBNET_MASK_TAG;
        pReqSpec->reqlist.list [pReqSpec->reqlist.len++] =
                                                        _DHCP_BRDCAST_ADDR_TAG;
        }

    /*
     * For now, create a separate watchdog timer for each lease.  Eventually,
     * a single watchdog timer will suffice for all leases, when combined
     * with a sorted queue containing relative firing times.  This enhancement
     * can wait.
     */

    /* Create event timer for state machine.  */

    pLeaseData->timer = wdCreate ();
    if (pLeaseData->timer == NULL)
        {
        free (pLeaseData);
        errno = S_dhcpcLib_MEM_ERROR;
        return (NULL);
        }

    /* Create signalling semaphore for completion of negotiation process.  */

    pLeaseData->leaseSem = semBCreate (SEM_Q_FIFO, SEM_EMPTY);
    if (pLeaseData->leaseSem == NULL)
        {
        wdDelete (pLeaseData->timer);
        free (pLeaseData);
        errno = S_dhcpcLib_MEM_ERROR;
        return (NULL);
        }

    /*
     * Manual leases will not reset the network interface unless specifically
     * requested to do so.  This lease type is replaced with DHCP_AUTOMATIC for
     * the lease established during system startup.  The parameters for
     * an automatic lease are always applied to the underlying interface.
     */

    pLeaseData->leaseType = DHCP_MANUAL;

    pLeaseData->initFlag = TRUE;

    /* Store the lease-specific data in the global list.  */

    dhcpcLeaseList [offset] = pLeaseData;

    return (pCookie);
    }

void *	dhcpc_FindCookieByIfName(char *if_name_P)
{
    int             i;
    LEASE_DATA *    pLeaseData;
    char            netDev [BOOT_DEV_LEN + 1];
	short			num_wan;

	SYSTEM_NumOfWanGet(&num_wan);
    for (i=0; i < num_wan; i++)
    {
        if (pDhcpcBootCookie[i] != NULL)
        {
            pLeaseData = (LEASE_DATA *)pDhcpcBootCookie[i];
            sprintf (netDev, "%s%d", pLeaseData->ifData.name,
                      pLeaseData->ifData.unit);
            if ( strcmp(netDev, if_name_P) == 0)
                return pDhcpcBootCookie[i];
        }
    }
    return NULL;
}


STATUS dhcpc_AddCookie(void *Cookie)
{
    int     i;
    STATUS  status = ERROR;
	short	num_wan;

	SYSTEM_NumOfWanGet(&num_wan);
    for (i=0; i<num_wan; i++)
    {
        if (pDhcpcBootCookie[i] == NULL)
        {
            pDhcpcBootCookie[i] = Cookie;
            status = OK;
            break;
        }
    }
    return status;
}

/*******************************************************************************
*
* dhcpcEventHookAdd - add a routine to handle configuration parameters
*
* This routine installs a hook routine to handle changes in the configuration
* parameters provided for the lease indicated by <pCookie>.  The hook provides
* an alternate configuration method for DHCP leases and uses the following
* interface:
* .CS
* void dhcpcEventHookRtn
*     (
*     int 	leaseEvent,	/@ new or expired parameters @/
*     void * 	pCookie 	/@ lease identifier from dhcpcInit() @/
*     )
* .CE
*
* The routine is called with the <leaseEvent> parameter set to DHCPC_LEASE_NEW
* whenever a lease is successfully established.  The DHCPC_LEASE_NEW event
* does not occur when a lease is renewed by the same DHCP server, since the
* parameters do not change in that case.  However, it does occur if the
* client rebinds to a different DHCP server.  The DHCPC_LEASE_INVALID event
* indicates that the configuration parameters for the corresponding lease may
* no longer be used.  That event occurs when a lease expires or a renewal
* or verification attempt fails, and coincides with re-entry into the initial
* state of the negotiation process.
*
* If the lease initialization specified automatic configuration of the
* corresponding network interface, any installed hook routine will be invoked
* after the new address information is applied.
*
* RETURNS: OK if notification hook added, or ERROR otherwise.
*
* ERRNO:
*  S_dhcpcLib_BAD_COOKIE
*  S_dhcpcLib_NOT_INITIALIZED
*/
STATUS dhcpcEventHookAdd
    (
    char        *if_name_P,     /* identifier */
    FUNCPTR     pEventHook      /* routine to handle lease parameters */
    )
    {
    LEASE_DATA *    pLeaseData;
    int             offset;

    /*
     * Use the cookie to access the lease-specific data structures.  For now,
     * just typecast the cookie.  This translation could be replaced with a more
     * sophisticated lookup at some point.
     */

    /* Find the cookie.*/
    pLeaseData =(LEASE_DATA *) dhcpc_FindCookieByIfName(if_name_P);

    for (offset = 0; offset < dhcpcMaxLeases; offset++)
        if (dhcpcLeaseList [offset] != NULL &&
                dhcpcLeaseList [offset] == pLeaseData)
            break;

    if (offset == dhcpcMaxLeases)
        {
        errno = S_dhcpcLib_BAD_COOKIE;
        return (ERROR);
        }

    if (!dhcpcInitialized || !pLeaseData->initFlag)
        {
        errno = S_dhcpcLib_NOT_INITIALIZED;
        return (ERROR);
        }

    /* Assign the event notification hook.  */

    pLeaseData->eventHookRtn = pEventHook;

    return (OK);
    }

/*******************************************************************************
*
* dhcpcEventHookDelete - remove the configuration parameters handler
*
* This routine removes the hook routine that handled changes in the
* configuration parameters for the lease indicated by <pCookie>.
* If the lease initialization specified automatic configuration of the
* corresponding network interface, the assigned address could change
* without warning after this routine is executed.
*
* RETURNS: OK if notification hook removed, or ERROR otherwise.
*
* ERRNO:
*  S_dhcpcLib_BAD_COOKIE
*  S_dhcpcLib_NOT_INITIALIZED
*
*/

STATUS dhcpcEventHookDelete
    (
    void * 	pCookie 	/* identifier returned by dhcpcInit() */
    )
    {
    LEASE_DATA * 	pLeaseData;
    int offset;

    /*
     * Use the cookie to access the lease-specific data structures. For now,
     * just typecast the cookie. This translation could be replaced with a more
     * sophisticated lookup at some point.
     */

    pLeaseData = (LEASE_DATA *)pCookie;

    for (offset = 0; offset < dhcpcMaxLeases; offset++)
        if (dhcpcLeaseList [offset] != NULL &&
                dhcpcLeaseList [offset] == pLeaseData)
            break;

    if (offset == dhcpcMaxLeases)
        {
        errno = S_dhcpcLib_BAD_COOKIE;
        return (ERROR);
        }

    if (!dhcpcInitialized || !pLeaseData->initFlag)
        {
        errno = S_dhcpcLib_NOT_INITIALIZED;
        return (ERROR);
        }

    /* Remove the event notification hook. */

    pLeaseData->eventHookRtn = NULL;

    return (OK);
    }

/*******************************************************************************
*
* dhcpcCacheHookAdd - add a routine to store and retrieve lease data
*
* This routine adds a hook routine that is called at the bound state
* (to store the lease data) and during the INIT_REBOOT state (to re-use the
* parameters if the lease is still active).  The calling sequence of the
* input hook routine is:
* .CS
* STATUS dhcpcCacheHookRtn
*     (
*     int command,			/@ requested cache operation @/
*     unsigned long *pTimeStamp,	/@ lease timestamp data @/
*     int *pDataLen,			/@ length of data to access @/
*     char *pBuffer			/@ pointer to data buffer @/
*     )
*
* .CE
* The hook routine should return OK if the requested operation is completed
* successfully, or ERROR otherwise.  All the supplied pointers reference memory
* locations that are reused upon return from the hook.  The hook routine
* must copy the data elsewhere.
*
* NOTE
* The setting of the cache hook routine during a dhcpcInit() call is
* recorded and used by the resulting lease throughout its lifetime.
* Since the hook routine is intended to store a single lease record,
* a separate hook routine should be specified before the dhcpcInit()
* call for each lease which will re-use its parameters across reboots.
*
* IMPLEMENTATION
* The <command> parameter specifies one of the following operations:
* .IP DHCP_CACHE_WRITE 25
* Save the indicated data.  The write operation must preserve the value
* referenced by <pTimeStamp> and the contents of <pBuffer>.  The <pDataLen>
* parameter indicates the number of bytes in that buffer.
* .IP DHCP_CACHE_READ
* Restore the saved data.  The read operation must copy the data from the
* most recent write operation into the location indicated by <pBuffer>,
* set the contents of <pDataLen> to the amount of data provided, and store
* the corresponding timestamp value in <pTimeStamp>.
* .IP
* The read operation has very specific requirements.  On entry, the value
* referenced by <pDataLen> indicates the maximum buffer size available at
* <pBuffer>.  If the amount of data stored by the previous write exceeds this
* value, the operation must return ERROR.  A read must also return ERROR if the
* saved timestamp value is 0.  Finally, the read operation must return ERROR if
* it is unable to retrieve all the data stored by the write operation or if the
* previous write was unsuccessful.
* .IP DHCP_CACHE_ERASE
* Ignore all stored data.  Following this operation, subsequent read operations
* must return ERROR until new data is written.  All parameters except
* <command> are NULL.
* .LP
*
* RETURNS: OK, always.
*
* ERRNO: N/A
*/

STATUS dhcpcCacheHookAdd
    (
    FUNCPTR 	pCacheHookRtn 	/* routine to store/retrieve lease data */
    )
    {
    dhcpcCacheHookRtn = pCacheHookRtn;
    return (OK);
    }

/*******************************************************************************
*
* dhcpcCacheHookDelete - delete a lease data storage routine
*
* This routine deletes the hook used to store lease data, preventing
* re-use of the configuration parameters across system reboots for
* all subsequent lease attempts.  Currently active leases will continue
* to use the routine specified before the lease initialization.
*
* RETURNS: OK, always.
*
* ERRNO: N/A
*/

STATUS dhcpcCacheHookDelete (void)
    {
    dhcpcCacheHookRtn = NULL;
    return (OK);
    }

/*******************************************************************************
*
* dhcpcOptionSet - add an option to the option request list
*
* This routine sets most client-to-server transmission options for the lease
* indicated by the <pCookie> parameter.  The <option> parameter specifies an
* option tag as defined in RFC 1533 and the updates published in the Internet
* Draft of November 1996.   For a listing of defined aliases for the known
* option tags, see dhcp/dhcp.h.  This routine cannot set the options associated
* with the following tags:
*
*     _DHCP_PAD_TAG
*     _DHCP_OPT_OVERLOAD_TAG
*     _DHCP_MSGTYPE_TAG
*     _DHCP_SERVER_ID_TAG
*     _DHCP_REQ_LIST_TAG
*     _DHCP_MAXMSGSIZE_TAG
*     _DHCP_END_TAG
*
* Most options only require specification of the appropriate tag in the
* <option> parameter.  In those cases, the dhcpcOptionSet() call adds the
* specified option tag to the option request list, if possible.  However, some
* options require additional information.  The tags for these options are:
*
*     _DHCP_VENDOR_SPEC_TAG
*     _DHCP_REQUEST_IPADDR_TAG
*     _DHCP_LEASE_TIME_TAG
*     _DHCP_ERRMSG_TAG
*     _DHCP_CLASS_ID_TAG
*     _DHCP_CLIENT_ID_TAG
*
* The _DHCP_LEASE_TIME_TAG and _DHCP_CLIENT_ID_TAG options each use the
* <value> parameter.  For _DHCP_LEASE_TIME_TAG, <value> specifies the desired
* lease length.  For _DHCP_CLIENT_ID_TAG, <value> specifies the type for a
* type/value pair.  No other options use this parameter.
*
* The _DHCP_VENDOR_SPEC_TAG, _DHCP_CLASS_ID_TAG and _DHCP_CLIENT_ID_TAG
* tags each require a value for the <length> parameter to specify the number
* of bytes of data provided.  No other options use this parameter.
*
* The <pData> parameter is relevant to the following option tags:
* .IP _DHCP_VENDOR_SPEC_TAG 25
* The <pData> parameter references a list of <length> bytes of options
* in the format specified by RFC 1533.
* .IP _DHCP_REQUEST_IPADDR_TAG
* The <pData> parameter indicates the string representation of the
* desired Internet address for the client in dot notation.
* .IP _DHCP_ERRMSG_TAG
* The <pData> parameter indicates the error message to send to the server
* when releasing the current IP address.  That location must be valid until
* the release is completed, since the message is not copied or stored in
* any way.
* .IP _DHCP_CLASS_ID_TAG
* The <pData> parameter references <length> bytes used as the value
* for the vendor class identifier.
* .IP _DHCP_CLIENT_ID_TAG
* The <pData> parameter references <length> bytes used as the value
* of a type/value pair.
* .IP
* The <pData> parameter is not used by any other options.
* .LP
*
* NOTE: With the exception of the _DHCP_ERRMSG_TAG option, the DHCP
* specification forbids changing options after a lease has been
* established.  Therefore, this routine should not be used after the
* dhcpcBind() call.  Changing any option other than the error message at
* that point could have unpredictable results.
*
* RETURNS: OK if the option was set successfully, or ERROR if the option
* is invalid or storage failed.
*
* ERRNO:
*  S_dhcpcLib_BAD_OPTION
*  S_dhcpcLib_OPTION_NOT_STORED
*
*/

STATUS dhcpcOptionSet
    (
    void * 	pCookie, 	/* identifier returned by dhcpcInit() */
    int 	option,		/* RFC 1533 tag of desired option */
    long 	value,		/* numeric value for option */
    long 	length,		/* length of data (if any) or 0 if unused  */
    char *	pData		/* option data, or NULL if none */
    )
    {
    LEASE_DATA *	 	pLeaseData;
    struct dhcp_reqspec * 	pReqSpec;

    u_long 	result;

    /*
     * Use the cookie to access the lease-specific data structures.  For now,
     * just typecast the cookie.  This translation could be replaced with a more
     * sophisticated lookup at some point.
     */

    pLeaseData = (LEASE_DATA *)pCookie;
    pReqSpec = &pLeaseData->leaseReqSpec;

    switch (option)          /* Check for restricted options.  */
        {
        case _DHCP_PAD_TAG: 		/* fall-through */
        case _DHCP_OPT_OVERLOAD_TAG: 	/* fall-through */
        case _DHCP_MSGTYPE_TAG: 	/* fall-through */
        case _DHCP_SERVER_ID_TAG: 	/* fall-through */
        case _DHCP_REQ_LIST_TAG: 	/* fall-through */
        case _DHCP_MAXMSGSIZE_TAG: 	/* fall-through */
        case _DHCP_END_TAG:
            errno = S_dhcpcLib_BAD_OPTION;
            return (ERROR);
            break;
        default:
            break;
        }

    if (option < 0 || option > _DHCP_LAST_OPTION)
        {
        errno = S_dhcpcLib_BAD_OPTION;
        return (ERROR);
        }

    if (pReqSpec->reqlist.len == _DHCP_MAX_OPTLEN)
        {
        switch (option)   /* Ignore error for options not in request list.  */
            {
            case _DHCP_VENDOR_SPEC_TAG: 	/* fall-through */
            case _DHCP_REQUEST_IPADDR_TAG: 	/* fall-through */
            case _DHCP_LEASE_TIME_TAG: 		/* fall-through */
            case _DHCP_ERRMSG_TAG: 		/* fall-through */
            case _DHCP_CLASS_ID_TAG: 		/* fall-through */
            case _DHCP_CLIENT_ID_TAG: 		/* fall-through */
            case _DHCP_HOSTNAME_TAG:
                break;
            default:         /* Request list is full.  */
                errno = S_dhcpcLib_OPTION_NOT_STORED;
                return (ERROR);
                break;
            }
        }

    switch (option)
        {
        case _DHCP_VENDOR_SPEC_TAG:
            if (length < 1 || length > _DHCP_MAX_OPTLEN)
                {
                errno = S_dhcpcLib_BAD_OPTION;
                return (ERROR);
                }
            if (pData == NULL)
                {
                errno = S_dhcpcLib_BAD_OPTION;
                return (ERROR);
                }
            bcopy (pData, pReqSpec->vendlist.list, (int)length);
            pReqSpec->vendlist.len = (u_char)length;
            break;
        case _DHCP_REQUEST_IPADDR_TAG:
            result = inet_addr (pData);
            if (result == ERROR)
                {
                errno = S_dhcpcLib_BAD_OPTION;
                return (ERROR);
                }
            pReqSpec->ipaddr.s_addr = result;
            break;
        case _DHCP_LEASE_TIME_TAG:
            pReqSpec->lease = value;
            break;
        case _DHCP_ERRMSG_TAG:
            pReqSpec->dhcp_errmsg = pData;
            break;
        case _DHCP_HOSTNAME_TAG:
            pReqSpec->hostName = pData;
            break;
        case _DHCP_CLASS_ID_TAG:
            if (length < 1 || length > _DHCP_MAX_OPTLEN)
                {
                errno = S_dhcpcLib_BAD_OPTION;
                return (ERROR);
                }
            if (pData == NULL)
                {
                errno = S_dhcpcLib_BAD_OPTION;
                return (ERROR);
                }
            pReqSpec->classId = calloc (1, sizeof (struct class_id));
            if (pReqSpec->classId == NULL)
                {
                errno = S_dhcpcLib_OPTION_NOT_STORED;
                return (ERROR);
                }
            pReqSpec->classId->id = calloc (length, sizeof (char));
            if (pReqSpec->classId->id == NULL)
                {
                errno = S_dhcpcLib_OPTION_NOT_STORED;
                return (ERROR);
                }
            bcopy (pData, pReqSpec->classId->id, (int)length);
            pReqSpec->classId->len = (u_char)length;
            break;
        case _DHCP_CLIENT_ID_TAG:
            if (length < 1 || length > _DHCP_MAX_OPTLEN)
                {
                errno = S_dhcpcLib_BAD_OPTION;
                return (ERROR);
                }
            if (pData == NULL)
                {
                errno = S_dhcpcLib_BAD_OPTION;
                return (ERROR);
                }
            pReqSpec->clid = calloc (1, sizeof (struct client_id));
            if (pReqSpec->clid == NULL)
                {
                errno = S_dhcpcLib_OPTION_NOT_STORED;
                return (ERROR);
                }
            pReqSpec->clid->id = calloc (length, sizeof (char));
            if (pReqSpec->clid->id == NULL)
                {
                errno = S_dhcpcLib_OPTION_NOT_STORED;
                return (ERROR);
                }
            bcopy (pData, pReqSpec->clid->id, (int)length);
            pReqSpec->clid->len = (unsigned char)length;
            pReqSpec->clid->type = (unsigned char)value;
            break;
        default:
            pReqSpec->reqlist.list [pReqSpec->reqlist.len++] = option;
            break;
        }
    return (OK);
    }

/*******************************************************************************
*
* dhcpcBind - obtain a set of network configuration parameters with DHCP
*
* This routine initiates a DHCP negotiation according to the process described
* in RFC 1541.  The <pCookie> argument contains the return value of an earlier
* dhcpcInit() call and is used to identify a particular lease.
*
* The <syncFlag> parameter specifies whether the DHCP negotiation started by
* this routine will execute synchronously or asynchronously.  An asynchronous
* execution will return after starting the DHCP negotiation, but a synchronous
* execution will only return once the negotiation process completes.
*
* When a new lease is established, any event hook provided for the lease
* will be called to process the configuration parameters.  The hook is also
* called when the lease expires or the negotiation process fails.  The results
* of an asynchronous DHCP negotiation are not available unless an event hook
* is installed.
*
* If automatic configuration of the underlying network interface was specified
* during the lease initialization, this routine will prevent all higher-level
* protocols from accessing the underlying network interface used during the
* initial lease negotiation until that process is complete.  In addition, any
* addressing information obtained will be applied to that network interface,
* which will remain disabled if the initial negotiation fails.  Finally, the
* interface will be disabled if the lease expires.
*
* NOTE
* If the DHCP client is used to obtain the VxWorks boot parameters, this
* routine is called automatically during system startup using the automatic
* reconfiguration.  Therefore, any calls to this routine which use the network
* boot device for message transfer when the DHCP client was used at boot time
* must not request automatic reconfiguration during initialization.  Otherwise,
* the resulting lease settings will conflict with the configuration maintained
* by the lease established during system startup.
*
* RETURNS: OK if routine completes, or ERROR otherwise.
*
* ERRNO:
*  S_dhcpcLib_BAD_COOKIE
*  S_dhcpcLib_NOT_INITIALIZED
*  S_dhcpcLib_BAD_OPTION
*
*/

STATUS dhcpcBind
    (
    void * 	pCookie, 	/* identifier returned by dhcpcInit() */
    BOOL 	syncFlag 	/* synchronous or asynchronous execution */
    )
    {
    int 		offset;
    STATUS		result = OK;
    LEASE_DATA * 	pLeaseData = NULL;

    /*
     * Use the cookie to access the lease-specific data structures.  For now,
     * just typecast the cookie.  This translation could be replaced with a more
     * sophisticated lookup at some point.
     */

    pLeaseData = (LEASE_DATA *)pCookie;

    for (offset = 0; offset < dhcpcMaxLeases; offset++)
        if (dhcpcLeaseList [offset] != NULL &&
                dhcpcLeaseList [offset] == pLeaseData)
            break;

    if (offset == dhcpcMaxLeases)
        {
        errno = S_dhcpcLib_BAD_COOKIE;
        return (ERROR);
        }

    if (!dhcpcInitialized || !pLeaseData->initFlag)
        {
        errno = S_dhcpcLib_NOT_INITIALIZED;
        return (ERROR);
        }

    if (syncFlag != TRUE && syncFlag != FALSE)
        {
        errno = S_dhcpcLib_BAD_OPTION;
        return (ERROR);
        }

    /* Examine type of DHCP lease to determine action required.  */

    if (pLeaseData->leaseType == DHCP_BOOTP)
        {
        return (OK);    /* BOOTP leases are always valid.  */
        }

    /* Wait for results if the startup lease is being renewed.  */

    if (pLeaseData->leaseType == DHCP_AUTOMATIC)
        pLeaseData->waitFlag = TRUE;
    else
        pLeaseData->waitFlag = syncFlag;

    if (pLeaseData->leaseType == DHCP_MANUAL && pLeaseData->leaseGood)
        {
        /* If redundant bind is requested, change to verification.  */

        result = dhcpcVerify (pCookie);
        return (result);
        }
    else	/* Obtain initial lease or renew startup lease.  */
        {
        dhcp_client (pCookie);    /* Perform bind process.  */

        /* Check results of synchronous lease attempt.  */

        if (pLeaseData->waitFlag)
            {
            pLeaseData->waitFlag = FALSE;    /* Disable further signals.  */
            semTake (dhcpcMutexSem, WAIT_FOREVER);
            if (pLeaseData->leaseGood)
                result = OK;
            else
                result = ERROR;
            semGive (dhcpcMutexSem);
            }
        else
            result = OK;
        }

    /*
     * If waitFlag was TRUE, the negotiation has completed.  Otherwise, it
     * has begun, and the installed event hook routine will be called at the
     * appropriate time.
     */

    return (result);
    }

/*******************************************************************************
*
* dhcpcVerify - renew an established lease
*
* This routine schedules the lease identified by the <pCookie> parameter
* for immediate renewal according to the process described in RFC 1541.
* If the renewal is unsuccessful, the lease negotiation process restarts.
* The routine is valid as long as the lease is currently active.  The
* routine is also called automatically in response to a dhcpcBind() call
* for an existing lease.
*
* NOTE
* This routine will disable the underlying network interface if the
* verification fails and automatic configuration was requested.  This may
* occur without warning if no event hook is installed.
*
* RETURNS: OK if verification scheduled, or ERROR otherwise.
*
* ERRNO:
*  S_dhcpcLib_BAD_COOKIE
*  S_dhcpcLib_NOT_INITIALIZED
*  S_dhcpcLib_NOT_BOUND
*
*/

STATUS dhcpcVerify
    (
    void * 	pCookie 	/* identifier returned by dhcpcInit() */
    )
    {
    int offset;
    LEASE_DATA * 	pLeaseData = NULL;
    STATUS result = OK;
    pLeaseData = (LEASE_DATA *)pCookie;

    for (offset = 0; offset < dhcpcMaxLeases; offset++)
        if (dhcpcLeaseList [offset] != NULL &&
                dhcpcLeaseList [offset] == pLeaseData)
            break;

    if (offset == dhcpcMaxLeases)
        {
        errno = S_dhcpcLib_BAD_COOKIE;
        return (ERROR);
        }

    if (!dhcpcInitialized || !pLeaseData->initFlag)
        {
        errno = S_dhcpcLib_NOT_INITIALIZED;
        return (ERROR);
        }

    /*
     * Use the cookie to access the lease-specific data structures.  For now,
     * just typecast the cookie.  This translation could be replaced with a more
     * sophisticated lookup at some point.
     */

    semTake (dhcpcMutexSem, WAIT_FOREVER);
    if (!pLeaseData->leaseGood)
        result = ERROR;
    semGive (dhcpcMutexSem);

    if (result == ERROR)
        {
        errno = S_dhcpcLib_NOT_BOUND;
        return (ERROR);
        }

    /* Construct and send a verification request to the client monitor task.  */

    result = dhcpcEventAdd (DHCP_USER_EVENT, DHCP_USER_VERIFY, pCookie, FALSE);

    return (result);
    }

/*******************************************************************************
*
* dhcpcRelease - relinquish specified lease
*
* This routine schedules the lease identified by the <pCookie> parameter
* for immediate release, regardless of time remaining, and removes all
* the associated data structures.  After the release completes, a new
* call to dhcpcInit() is required before attempting another lease.
*
* NOTE
* This routine will disable the underlying network interface if automatic
* configuration was requested.  This may occur without warning if no event
* hook is installed.
*
* RETURNS: OK if release scheduled, or ERROR otherwise.
*
* ERRNO:
*  S_dhcpcLib_BAD_COOKIE
*  S_dhcpcLib_NOT_INITIALIZED
*
*/

STATUS dhcpcRelease
    (
    void * 	pCookie 	/* identifier returned by dhcpcInit() */
    )
    {
    int offset;
    LEASE_DATA *        pLeaseData = NULL;
    STATUS result = OK;

    pLeaseData = (LEASE_DATA *)pCookie;

    for (offset = 0; offset < dhcpcMaxLeases; offset++)
        if (dhcpcLeaseList [offset] != NULL &&
                dhcpcLeaseList [offset] == pLeaseData)
            break;

    if (offset == dhcpcMaxLeases)
        {
        errno = S_dhcpcLib_BAD_COOKIE;
        return (ERROR);
        }

    if (!dhcpcInitialized || !pLeaseData->initFlag)
        {
        errno = S_dhcpcLib_NOT_INITIALIZED;
        return (ERROR);
        }

    /*
     * Use the cookie to access the lease-specific data structures.  For now,
     * just typecast the cookie.  This translation could be replaced with a more
     * sophisticated lookup at some point.
     */

    /* Construct and send a release request to the client monitor task.  */

    result = dhcpcEventAdd (DHCP_USER_EVENT, DHCP_USER_RELEASE, pCookie, FALSE);

    return (result);
    }

/*******************************************************************************
*
* dhcpcShutdown - disable DHCP client library
*
* This routine schedules the lease monitor task to clean up memory and exit,
* after releasing all currently active leases.  The network boot device
* will be disabled if the DHCP client was used to obtain the VxWorks boot
* parameters and the resulting lease is still active.  Any other interfaces
* using the addressing information from leases set for automatic configuration
* will also be disabled.  Notification of a disabled interface will not occur
* unless an event hook has been installed.  After the processing started by
* this request completes, the DHCP client library is unavailable until
* restarted with the dhcpcLibInit() routine.
*
* RETURNS: OK if shutdown scheduled, or ERROR otherwise.
*
* ERRNO:
*  S_dhcpcLib_NOT_INITIALIZED
*
*/

STATUS dhcpcShutdown (void)
    {
    STATUS 	result;

    if (!dhcpcInitialized)
        {
        errno = S_dhcpcLib_NOT_INITIALIZED;
        return (ERROR);
        }

    /* Construct and send a shutdown request to the client monitor task.  */

    result = dhcpcEventAdd (DHCP_USER_EVENT, DHCP_USER_SHUTDOWN, NULL, FALSE);

    return (result);
    }

/*******************************************************************************
*
* dhcpcOptionGet - retrieve an option provided to a client and store in a buffer
*
* This routine retrieves the data for the specified <option>, if present
* for the lease indicated by <pCookie>.  The data is stored in the provided
* buffer, whose length must be specified.  If the <option> is found, the amount
* of data available is stored in the location referenced by the <pLength>
* parameter.  The option is not available if the DHCP client is not in the
* bound state or if the server did not provide it.  After returning, the
* provided buffer may contain IP addresses stored in network byte order.  All
* other numeric values are stored in host byte order.  See RFC 1533 for specific
* details on the data retrieved.
*
* RETURNS: OK if option available, or ERROR otherwise.
*
* ERRNO:
*  S_dhcpcLib_BAD_COOKIE
*  S_dhcpcLib_NOT_INITIALIZED
*  S_dhcpcLib_NOT_BOUND
*  S_dhcpcLib_OPTION_NOT_PRESENT
*
* SEE ALSO
* dhcpcOptionSet()
*/

STATUS dhcpcOptionGet
    (
    void * 	pCookie, 	/* identifier returned by dhcpcInit() */
    int 	option,		/* RFC 1533 option tag */
    int * 	pLength, 	/* size of provided buffer and data returned */
    char *	pBuf 		/* location for option data */
    )
    {
    char *	pData;
    int  	amount;
    int 	limit;
    int 	offset;
    LEASE_DATA * 	pLeaseData;

    limit = *pLength;

    /*
     * Use the cookie to access the lease-specific data structures.  For now,
     * just typecast the cookie.  This translation could be replaced with a more
     * sophisticated lookup at some point.
     */

    pLeaseData = (LEASE_DATA *)pCookie;

    for (offset = 0; offset < dhcpcMaxLeases; offset++)
        if (dhcpcLeaseList [offset] != NULL &&
                dhcpcLeaseList [offset] == pLeaseData)
            break;

    if (offset == dhcpcMaxLeases)
        {
        errno = S_dhcpcLib_BAD_COOKIE;
        return (ERROR);
        }

    if (!dhcpcInitialized || !pLeaseData->initFlag)
        {
        errno = S_dhcpcLib_NOT_INITIALIZED;
        return (ERROR);
        }

    /* Check if client is bound.  */

    semTake (dhcpcMutexSem, WAIT_FOREVER);
    if (!pLeaseData->leaseGood)
        {
        errno = S_dhcpcLib_NOT_BOUND;
        semGive (dhcpcMutexSem);
        return (ERROR);
        }
    semGive (dhcpcMutexSem);

    /* Check if requested parameter is available.  */

    if (!ISSET (pLeaseData->dhcpcParam->got_option, option))
        {
        errno = S_dhcpcLib_OPTION_NOT_PRESENT;
        return (ERROR);
        }

    /* Find the location of option data */

    pData = dhcpcOptionFind (pLeaseData, option, &amount);
    if (pData == NULL)
        {
        errno = S_dhcpcLib_OPTION_NOT_PRESENT;
        return (ERROR);
        }

    if (amount == 0)
        {
        /* Requested parameter not provided by server - no data returned.  */

        *pLength = 0;
        return (OK);
        }

    if (amount < limit)         /* Adjust for size of option data.  */
        limit = amount;

    bcopy (pData, pBuf, limit);
    *pLength = limit;

    return (OK);
    }

/*******************************************************************************
*
* dhcpcServerGet - retrieve the current DHCP server
*
* This routine returns the DHCP server that supplied the configuration
* parameters for the lease specified by the <pCookie> argument.  This
* information is available only if the lease is in the bound state.
*
* RETURNS: OK if in bound state and server available, or ERROR otherwise.
*
* ERRNO:
*  S_dhcpcLib_BAD_COOKIE
*  S_dhcpcLib_NOT_INITIALIZED
*  S_dhcpcLib_NOT_BOUND
*/

STATUS dhcpcServerGet
    (
    void * 		pCookie, 	/* identifier returned by dhcpcInit() */
    struct in_addr * 	pServerAddr 	/* location for address of server */
    )
    {
    int offset;
    LEASE_DATA * 	pLeaseData;
    STATUS 		result = OK;

    /*
     * Use the cookie to access the lease-specific data structures.  For now,
     * just typecast the cookie.  This translation could be replaced with a more
     * sophisticated lookup at some point.
     */

    pLeaseData = (LEASE_DATA *)pCookie;

    for (offset = 0; offset < dhcpcMaxLeases; offset++)
        if (dhcpcLeaseList [offset] != NULL &&
                dhcpcLeaseList [offset] == pLeaseData)
            break;

    if (offset == dhcpcMaxLeases)
        {
        errno = S_dhcpcLib_BAD_COOKIE;
        return (ERROR);
        }

    if (!dhcpcInitialized || !pLeaseData->initFlag)
        {
        errno = S_dhcpcLib_NOT_INITIALIZED;
        return (ERROR);
        }

    semTake (dhcpcMutexSem, WAIT_FOREVER);
    if (!pLeaseData->leaseGood)
        result = ERROR;
    semGive (dhcpcMutexSem);

    if (result == ERROR)
        {
        errno = S_dhcpcLib_NOT_BOUND;
        return (ERROR);
        }

    if (pLeaseData->dhcpcParam->server_id.s_addr != 0)
        bcopy ( (char *)&pLeaseData->dhcpcParam->server_id,
               (char *)pServerAddr, sizeof (struct in_addr));
    else
        result = ERROR;

    return (result);
    }

/*******************************************************************************
*
* dhcpcTimerGet - retrieve current lease timers
*
* This routine returns the number of clock ticks remaining on the timers
* governing the DHCP lease specified by the <pCookie> argument.  This
* information is only available if the lease is in the bound state.
* Therefore, this routine will return ERROR if a BOOTP reply was accepted.
*
* RETURNS: OK if in bound state and values available, or ERROR otherwise.
*
* ERRNO:
*  S_dhcpcLib_BAD_COOKIE
*  S_dhcpcLib_NOT_INITIALIZED
*  S_dhcpcLib_NOT_BOUND
*  S_dhcpcLib_TIMER_ERROR
*/

STATUS dhcpcTimerGet
    (
    void * 	pCookie, 	/* identifier returned by dhcpcInit() */
    int * 	pT1, 		/* time until lease renewal */
    int * 	pT2 		/* time until lease rebinding */
    )
    {
    int 	offset;
    time_t	current = 0;
    long	timer1;
    long	timer2;
    LEASE_DATA *        pLeaseData;
    STATUS 		result = OK;

    /*
     * Use the cookie to access the lease-specific data structures.  For now,
     * just typecast the cookie.  This translation could be replaced with a more
     * sophisticated lookup at some point.
     */

    pLeaseData = (LEASE_DATA *)pCookie;

    for (offset = 0; offset < dhcpcMaxLeases; offset++)
        if (dhcpcLeaseList [offset] != NULL &&
                dhcpcLeaseList [offset] == pLeaseData)
            break;

    if (offset == dhcpcMaxLeases)
        {
        errno = S_dhcpcLib_BAD_COOKIE;
        return (ERROR);
        }

    if (!dhcpcInitialized || !pLeaseData->initFlag)
        {
        errno = S_dhcpcLib_NOT_INITIALIZED;
        return (ERROR);
        }

    if (pLeaseData->leaseType == DHCP_BOOTP)
        {
#ifdef DHCPC_DEBUG
        logMsg ("No timer values: BOOTP reply accepted.\n", 0, 0, 0, 0, 0, 0);
#endif
        errno = S_dhcpcLib_OPTION_NOT_PRESENT;
        return (ERROR);
        }

    if (time (&current) == -1)
        {
#ifdef DHCPC_DEBUG
        logMsg ("time() error in dhcpTimerGet().\n", 0, 0, 0, 0, 0, 0);
#endif
        errno = S_dhcpcLib_TIMER_ERROR;
        return (ERROR);
        }

    semTake (dhcpcMutexSem, WAIT_FOREVER);
    if (!pLeaseData->leaseGood)
        result = ERROR;
    semGive (dhcpcMutexSem);

    if (result == ERROR)
        {
        errno = S_dhcpcLib_NOT_BOUND;
        return (ERROR);
        }

    timer1 = pLeaseData->dhcpcParam->lease_origin +
                 pLeaseData->dhcpcParam->dhcp_t1 - current;
    timer2 = pLeaseData->dhcpcParam->lease_origin +
                 pLeaseData->dhcpcParam->dhcp_t2 - current;

    if (timer1 < 0)
        timer1 = 0;
    if (timer2 < 0)
        timer2 = 0;

    *pT1 = timer1 * sysClkRateGet();
    *pT2 = timer2 * sysClkRateGet();

    return (OK);
    }

/*******************************************************************************
*
* dhcpcParamsGet - retrieve current configuration parameters
*
* This routine copies the current configuration parameters for the lease
* specified by the <pCookie> argument to the user-supplied structure.  That
* structure, defined in dhcp/dhcpc.h, should contain non-NULL pointers to
* indicate the parameters of interest.  All other values within the structure
* must be set to 0 before calling the routine.  The requested information is
* only retrieved if the specified lease is in the bound state and knows that
* its parameters are good.
*
* Many of the parameters within the user-supplied structure use one of the
* following secondary data types: struct in_addrs, struct u_shorts, and
* struct vendor_list.  Each of those structures accepts a length designation
* and a data pointer.  For the first two data types, the 'num' member indicates
* the size of the buffer in terms of the number of underlying elements.  For
* example, the STATIC_ROUTE option returns one or more IP address pairs.  So,
* setting the 'num' member to 2 in the static_route entry would indicate that
* the corresponding buffer contained 16 bytes.  By contrast, the 'len' member
* in the struct 'vendor_list' data type consists of the buffer size, in bytes.
* See RFC 1533 for specific details on the types of data for each option.
*
* On return, each of the length designators are set to indicate the amount of
* data returned.  For instance, the 'num' member in the static_route entry could
* be set to 1 to indicate that only one IP address pair of 8 bytes was
* available.
*
* RETURNS: OK if in bound state, or ERROR otherwise.
*
* ERRNO:
*  S_dhcpcLib_NOT_INITIALIZED
*  S_dhcpcLib_NOT_BOUND
*
*/

STATUS dhcpcParamsGet
    (
    void * 	pCookie, 	/* identifier returned by dhcpcInit() */
    struct dhcp_param *	pParamList	/* requested parameters */
    )
    {
    int offset;
    LEASE_DATA *        pLeaseData;
    STATUS 		result = OK;

    /*
     * Use the cookie to access the lease-specific data structures.  For now,
     * just typecast the cookie.  This translation could be replaced with a more
     * sophisticated lookup at some point.
     */

    pLeaseData = (LEASE_DATA *)pCookie;

    for (offset = 0; offset < dhcpcMaxLeases; offset++)
        if (dhcpcLeaseList [offset] != NULL &&
                dhcpcLeaseList [offset] == pLeaseData)
            break;

    if (offset == dhcpcMaxLeases)
        {
        errno = S_dhcpcLib_BAD_COOKIE;
        return (ERROR);
        }

    if (!dhcpcInitialized || !pLeaseData->initFlag)
        {
        errno = S_dhcpcLib_NOT_INITIALIZED;
        return (ERROR);
        }

    semTake (dhcpcMutexSem, WAIT_FOREVER);
    if (!pLeaseData->leaseGood)
        result = ERROR;
    semGive (dhcpcMutexSem);

    if (result == ERROR)
        {
        errno = S_dhcpcLib_NOT_BOUND;
        return (ERROR);
        }

    dhcpcParamsCopy (pLeaseData, pParamList);

    return (OK);
    }

