/****************************************************************************
*
*   Name:           SnmpAIfTable.h
*
*   Description:    SNMP Interfaces Table header file
*
*   Copyright:      (c) 2002 Conexant Systems Inc.
*
*****************************************************************************
*   $Author: shao-gh $
*   $Revision: 1.1.1.1 $
*   $Modtime:   Apr 15 2003 06:55:04  $
****************************************************************************/

#ifndef _SNMPA_IF_TABLE_
#define _SNMPA_IF_TABLE_

#ifdef __cplusplus
extern "C" {
#endif

#include <vxWorks.h>
#include <bsptypes.h>

#include "SnmpApi.h"
#include "SnmpAIfMib.h"

/******************************************************************************
|
|  Function:    SnmpaIfIndexGetByIfTypeIndex
|
|  Description: Get the ifIndex of the specified interface
|
|  Parameters:  pIfTableHandle  the IfTable to search
|               IfType          the interface type to get
|               TypeIndex       the index of the specified interface type
|               pIfIndex        pointer to store the interface index
|
|  Returns:     TRUE        the first interface is found
|               FALSE       the first interface is not found
|
*******************************************************************************/
BOOL SnmpaIfIndexGetByIfTypeIndex
(
    void*   pIfTableHandle, 
    UINT32  IfType, 
    UINT32  TypeIndex, 
    UINT32  *pIfIndex
);

/******************************************************************************
|
|  Function:    SnmpaIfTypeIndexGetByIfIndex
|
|  Description: Get the IfType and TypeIndex of the specified interface
|
|  Parameters:  pIfTableHandle  the IfTable to search
|               IfIndex         the interface index to get
|               pIfType         pointer to store the interface type retrived
|               pTypeIndex      pointer to store the index retrieved
|
|  Returns:     TRUE        the first interface is found
|               FALSE       the first interface is not found
|
*******************************************************************************/
BOOL SnmpaIfTypeIndexGetByIfIndex
(
    void*   pIfTableHandle, 
    UINT32  IfIndex, 
    UINT32* pIfType, 
    UINT32* pTypeIndex
);

/******************************************************************************
|
|  Function:    SnmpaIfCreate
|
|  Description: Create an entry in the specified IfTable.
|
|  Parameters:  pIfTableHandle  the IfTable to search
|               IfType          the interface type to get
|               TypeIndex       the index of the specified interface type
|               pIfIndex        pointer to store the newly assigned interface index
|
|  Returns:     TRUE        the entry is created and ifIndex is stored in *pIfIndex
|               FALSE       failed to create the entry
|
*******************************************************************************/
BOOL SnmpaIfCreate
(
    void*   pIfTableHandle, 
    UINT32  IfType, 
    UINT32  TypeIndex, 
    UINT32* pIfIndex
);

/******************************************************************************
|
|  Function:    SnmpaIfDestroy
|
|  Description: Destroy an entry in the specified IfTable.
|               The entry is marked not present and is not physically deleted.
|
|  Parameters:  pIfTableHandle  the IfTable to search
|               IfIndex         the interface to destroy
|
|  Returns:     TRUE        the entry is destroyed
|               FALSE       failed to destroy the entry
|
*******************************************************************************/
BOOL SnmpaIfDestroy(void* pIfTableHandle, UINT32 IfIndex);

/******************************************************************************
|
|  Function:    SnmpaIfRegister
|
|  Description: Register the presence of an interface with the callback function.
|
|  Parameters:  IfType      Interface type
|               IfInstance  Index of this interface type
|               pFunc       callback function to the the interface MIB
|
|  Returns:     TRUE        the interface is registered
|               FALSE       failed to register the interface
|
*******************************************************************************/
BOOL SnmpaIfRegister
(
    UINT32  IfType, 
    UINT32  IfInstance, 
    FUNCPTR pFunc
);

/******************************************************************************
|
|  Function:    SnmpaIfRegister
|
|  Description: Unregister the presence of an interface with the callback function.
|
|  Parameters:  IfType      Interface type
|               IfInstance  Index of this interface type
|
|  Returns:     TRUE        the interface is unregistered
|               FALSE       failed to unregister the interface
|
*******************************************************************************/
BOOL SnmpaIfUnregister
(
    UINT32  IfType, 
    UINT32  IfInstance
);

/******************************************************************************
|
|  Function:    SnmpaIfTableCreate
|
|  Description: Create an Interface Table and, if specified, duplicate all data
|               from the current SNMP Interface Table.
|
|  Parameters:  bDuplicate  TRUE: duplicate data from current SNMP IfTable
|                           FALSE: create an empty IfTable only
|
|  Returns:     pIfTable    IfTable handle created
|                           NULL: failed to create IfTable as requested
|
*******************************************************************************/
void* SnmpaIfTableCreate(BOOL bDuplicate);

/******************************************************************************
|
|  Function:    SnmpaIfTableDestroy
|
|  Description: Destroy the specified Interface Table.
|
|  Parameters:  pIfTableHandle  the IfTable to destroy
|
|  Returns:     TRUE    IfTable destroyed
|               FALSE   Failed to destory IfTable
|
*******************************************************************************/
BOOL SnmpaIfTableDestroy(void* pIfTableHandle);

/******************************************************************************
|
| Function:     MIB_IfTableInit
|
| Description:  Initialize the Interface Table & add its MIB subtree.
|               Must be called before any other module calls us to register 
|               interfaces.
|
| Parameters:   none
|
| Returns:      none
|
*******************************************************************************/
void MIB_IfTableInit(void);

/******************************************************************************
|
|  Function:    MIB_IsIfPresent
|
|  Description: Check if the specified interface is present.
|
|  Parameters:  IfIndex     interface index
|
|  Returns:     TRUE        the interface is present
|               FALSE       the interface is not present
|
*******************************************************************************/
BOOL MIB_IsIfPresent (UINT32 IfIndex);

/******************************************************************************
|
|  Function:    MIB_GetFirstIfIndex
|
|  Description: Get the index of the first present interface in the SNMP interface table
|
|  Parameters:  pIfIndex    pointer to store the first interface index
|               IfType      the interface type to get
|                           0 - any interface type
|
|  Returns:     TRUE        the first interface is found
|               FALSE       the first interface is not found
|
*******************************************************************************/
BOOL MIB_GetFirstIfIndex(m_UINT32 *pIfIndex, UINT32 IfType);

/******************************************************************************
|
|  Function:    MIB_GetNextIfIndex
|
|  Description: Get the index of the next present interface in the SNMP interface table
|
|  Parameters:  pIfIndex    pointer to the current interface index
|               IfType      the interface type to get
|                           0 - any interface type
|
|  Returns:     TRUE        the next interface is found
|               FALSE       the next interface is not found
|
*******************************************************************************/
BOOL MIB_GetNextIfIndex(m_UINT32 *pIfIndex, UINT32 IfType);

/******************************************************************************
|
|  Function:    MIB_GetIfNumber
|
|  Description: Return total number of interfaces present in the interface table.
|
|  Parameters:  none
|
|  Returns:     g_ifNumber  total # interfaces present
|
*******************************************************************************/
UINT32 MIB_GetIfNumber(void);

/******************************************************************************
|
|  Function:    MIB_IfIndexType2Instance
|
|  Description: Validate the interface (with the specified interface index) has
|               the specified interface type.  If it is, return the instance
|               number of that interface for that interface type.
|
|  Parameters:  IfIndex     interface index
|               IfType      interface type
|               pIfInstance pointer to the instance #
|
|  Returns:     TRUE        the specified interface is the specified interface type
|                           store the instance# in *pIfInstance
|               FALSE       the specified interface is NOT the specified interface type
|
*******************************************************************************/
BOOL MIB_IfIndexType2Instance(UINT32 IfIndex, UINT32 IfType, UINT32 *pIfInstance);

/******************************************************************************
|
|  Function:    MIB_IfTypeInstance2Index
|
|  Description: Map interface type and instance to interface index.
|
|  Parameters:  IfType      interface type
|               TypeIndex  instance #
|               pIfIndex    pointer to interface index
|
|  Returns:     TRUE        the specified interface is found
|               FALSE       the specified interface is NOT found
|
*******************************************************************************/
BOOL MIB_IfTypeInstance2Index(UINT32 IfType, UINT32 TypeIndex, UINT32 *pIfIndex);

/******************************************************************************
|
|  Function:    MIB_GetIfEntry
|
|  Description: This function retrieves the interface MIBs specified in
|               pData->ifIndex by calling the appropriate SNMP Interface MIB API.
|
|  Parameters:  pData   pointer to the memory to store the retrieved data
|                       NOTE: pData->ifIndex must be filled with the desired
|                             interface index
|
|  Returns:     OK      MIBs are retrieved successfully
|               ERROR   Can't find the specified interface
|
*******************************************************************************/
STATUS MIB_GetIfEntry(MIB_INTERFACE_ENTRY *pData);


#ifdef __cplusplus
}
#endif

#endif /* _SNMPA_IF_TABLE_ */

/*       
 *=========================================================================+
 * Revision History:
 *
 * $Log: SnmpAIfTable.h,v $
 * Revision 1.1.1.1  2004/04/01 09:10:04  shao-gh
 * no message
 *
 * Revision 1.1.1.1  2004/01/16 09:32:58  SHAO-GH
 * no message
 *
 * Revision 1.1.1.1  2003/12/23 10:28:48  SHAO-GH
 * no message
 *
 * Revision 1.1.1.1  2003/11/21 08:29:14  SHAO-GH
 * 2003.11.20 Shao Guohua use win cvs first time from sourcesafe 
 *
 * 
 *    Rev 1.2   Apr 16 2003 18:45:42   liniy
 * Provide APIs to build/destory IF Table.
 * 
 *    Rev 1.1   Oct 11 2002 09:07:38   liniy
 * Exclude objects private to SnmpAIfTable.c.
 * 
 *    Rev 1.0   Oct 09 2002 11:01:00   liniy
 *  
 * 
 *    Rev 1.0   Nov 09 2001 11:50:24   liniy
 *  
 * 
 *    Rev 1.0   Nov 06 2001 16:00:54   liniy
 *  
 * 
 *=========================================================================+
 */

