/*
 * Copyright (C) 2005 by egnite Software GmbH
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

/*!
 * \file arm7tdmi.c
 * \brief Target dependant ICE routines for the ARM7TDMI CPU.
 *
 * \verbatim
 *
 * $Log: arm7tdmi.c,v $
 * Revision 1.3  2006/03/21 16:36:52  haraldkipp
 * Test successfully on Linux with Wiggler clone.
 *
 * Revision 1.2  2006/03/20 14:13:02  haraldkipp
 * RAM upload speed increased by factor 8.
 * Missing IceUpdate() dummy added.
 *
 * Revision 1.1.1.1  2005/09/14 09:01:08  haraldkipp
 * Initial import.
 *
 *
 * \endverbatim
 */

#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include "tap.h"
#include "ice.h"

#if defined(__linux__) || defined(__FreeBSD__)
#define stricmp strcasecmp
#endif

/*! Selects the scan path register. */
#define TAP_SCAN_N         0x02
/*! Returns core to system state after Run-Test/Idle. */
#define TAP_RESTART        0x04
/*! Places the previously selected scan chain in core test mode. */
#define TAP_INTEST         0x0C
/*! Selects the ID register. */
#define TAP_IDCODE         0x0E

#define ICE_DBG_CTRL        0
#define ICE_DBG_STAT        1
#define ICE_DCC_CTRL        4
#define ICE_DCC_DATA        5
#define ICE_WP0_ADDR        8
#define ICE_WP0_ADDR_MASK   9
#define ICE_WP0_DATA        10
#define ICE_WP0_DATA_MASK   11
#define ICE_WP0_CTRL        12
#define ICE_WP0_CTRL_MASK   13
#define ICE_WP1_ADDR        16
#define ICE_WP1_ADDR_MASK   17
#define ICE_WP1_DATA        18
#define ICE_WP1_DATA_MASK   19
#define ICE_WP1_CTRL        20
#define ICE_WP1_CTRL_MASK   21


#define DCC_HOST_TXBSY   1
#define DCC_HOST_RXRDY   2

/* ARM instructions */

#define ARM_NOP             0xE1A00000  /* MOV R0, R0 */

#define ARM_REG_SP      13
#define ARM_REG_LR      14
#define ARM_REG_PC      15
#define ARM_REG_CPSR    16

static unsigned long ice_dsr;
static unsigned long arm_regs[17];
static char *arm_regnames[17] = {
    "R0", "R1", "R2", "R3", "R4", "R5", "R6", "R7",
    "R8", "R9", "R10", "R11", "R12", "SP", "LR", "PC",
    "CPSR"
};

/*
 * \brief Reverse the order of a specified number of bits in a long value.
 *
 * \param val Value to convert.
 * \param num Number of bits starting with the LSB.
 *
 * \return Converted value.
 */
static unsigned long ReverseBitOrder(unsigned long val, int num)
{
    unsigned long rc = 0;

    while (num--) {
        rc <<= 1;
        rc |= val & 1;
        val >>= 1;
    }
    return rc;
}

/*! 
 * \brief Select an ARM7 scan chain.
 * 
 * On return the selected chain will be in test mode.
 *
 * \param sc Number of the scan chain to select. Scan chain 0 allows 
 *           access to the entire peripherie of the core. Scan chain 1
 *           is a subset of scan chain 0 and allows access to the data 
 *           bus and the breakpoint signal. The EmbeddedICE-RT logic
 *           can be accessed via scan chain 2.
 */
void Arm7SelectScanChain(unsigned char sc)
{
    static unsigned char current = 255;

    if (sc != current) {
        if (sc != 255) {
            /* Write the number of the scan chain into the scan path 
               select register. */
            TapInstruction(TAP_SCAN_N);
            TapData(NULL, &sc, 4);

            /* Put the selected scan chain in test mode. */
            TapInstruction(TAP_INTEST);
        }
        current = sc;
    }
}

/*!
 * \brief Access the core data bus and the BREAKPT signal.
 *
 * \param data  Contains the value to put on the data bus.
 * \param brkpt If set to 1, the next instruction will be executed at 
 *              system speed (MCLK).
 *
 * \return The value retrieved from the data bus.
 */
static unsigned long Arm7ScanChain1(unsigned long data, int brkpt)
{
    unsigned long rc;
    unsigned char wbuf[5];
    unsigned char rbuf[5];

    data = ReverseBitOrder(data, 32);
    wbuf[0] = (unsigned char) ((data << 1) | brkpt);
    wbuf[1] = (unsigned char) (data >> 7);
    wbuf[2] = (unsigned char) (data >> 15);
    wbuf[3] = (unsigned char) (data >> 23);
    wbuf[4] = (unsigned char) (data >> 31);

    /* Select the Debug Scan Chain */
    Arm7SelectScanChain(1);
    TapData(rbuf, wbuf, 33);

    rc = ((unsigned long) (rbuf[0]) >> 1) +
        ((unsigned long) (rbuf[1]) << 7) +
        ((unsigned long) (rbuf[2]) << 15) + 
        ((unsigned long) (rbuf[3]) << 23) + 
        ((unsigned long) (rbuf[4]) << 31);

    TapNext(0);                 /* -> Run-Test/Idle for DCLK */

    return ReverseBitOrder(rc, 32);
}

/*!
 * \brief Access an EmbeddedICE-RT macro register.
 *
 * \param reg  Register number including the optional write flag (bit 5).
 * \param data Data to write into the macrocell register. Ignored if bit 5
 *             of the reg parameter is not set.
 *
 * \return If bit 5 of the reg parameter is not set, then the contents of
 *         the specified macrocell register is returned. Otherwise the
 *         returned value should be ignored.
 */
unsigned long Arm7ScanChain2(int reg, unsigned long data)
{
    unsigned char buf[5];

    buf[0] = (unsigned char) data;
    buf[1] = (unsigned char) (data >> 8);
    buf[2] = (unsigned char) (data >> 16);
    buf[3] = (unsigned char) (data >> 24);
    buf[4] = (unsigned char) (reg);

    Arm7SelectScanChain(2);

    TapData(NULL, buf, 38);
    if (reg & 0x20) {
        return data;
    }
    TapData(buf, NULL, 38);

    return (unsigned long) (buf[0]) +
        ((unsigned long) (buf[1]) << 8) + 
        ((unsigned long) (buf[2]) << 16) + 
        ((unsigned long) (buf[3]) << 24);
}

/*!
 * \brief Execute a single instruction in test mode.
 *
 * Additional NOPs are pushed into the instruction pipeline to
 * move the instruction into the execute stage.
 *
 * \param opcode The instruction code to execute.
 */
static void Arm7TestExec(unsigned long opcode)
{
    /* Push instruction into the pipeline. */
    Arm7ScanChain1(opcode, 0);
    /* Push a NOP to move instruction into the decode stage. */
    Arm7ScanChain1(ARM_NOP, 0);
    /* Push a NOP to move instruction into the execute stage. */
    Arm7ScanChain1(ARM_NOP, 0);
}

/*!
 * \brief Set a given number of ARM CPU registers.
 *
 * \param regs The number of registers to set starting from R0.
 * \param tab  Pointer to an array containing the values to set.
 */
static void Arm7WriteRegs(int regs, unsigned long *tab)
{
    int i;
    unsigned long mask = 1;

    /* Set LDM mask. */
    for (i = regs; i; i--) {
        mask |= (1 << (regs - i));
    }

    /* Execute LDMIA LR, {R0-Ri}. */
    Arm7TestExec(0xE89E0000 | mask);

    /* Put the values on the bus. */
    for (i = 0; i < regs; i++, tab++) {
        Arm7ScanChain1(*tab, 0);
    }

    /* This cycle stores the data bus value in the registers, */
    Arm7ScanChain1(ARM_NOP, 0);

    /* The program counter requires two additional cycles. */
    if (mask & 0x8000) {
        Arm7ScanChain1(ARM_NOP, 0);
        Arm7ScanChain1(ARM_NOP, 0);
    }
}

/*!
 * \brief Read CPU register contents.
 *
 * \param reg  Register index.
 *
 * \return Contents of the specified register.
 */
static unsigned long Arm7ReadReg(unsigned char reg)
{
    /* STR R[index], [R14]. Bits 12..15 specify the index. */
    Arm7TestExec(0xE58E0000 | ((unsigned long) reg << 12));

    /* Get data from the bus. */
    return Arm7ScanChain1(ARM_NOP, 0);
}

/*!
 * \brief Save ARM CPU registers.
 *
 * \todo Currently only R0-R15 and CPSR are saved.
 */
static void Arm7SaveContext(void)
{
    int i;

    /* Execute STMIA R0, {R0-R15}. */
    Arm7TestExec(0xE880FFFF);

    /* Get register contents from the bus. */
    for (i = 0; i < 16; i++) {
        arm_regs[i] = Arm7ScanChain1(ARM_NOP, 0);
    }

    /* Correct PC contents, 3 debug instruction + pipeline. */
    arm_regs[ARM_REG_PC] -= 5 * 4;

    /* Execute MRS R0, CPSR */
    Arm7TestExec(0xE10F0000);

    /* Read the result. */
    arm_regs[ARM_REG_CPSR] = Arm7ReadReg(0);
}

/*!
 * \brief Set CPU register contents.
 *
 * \param reg  Register index.
 * \param data Value to store into the specified register.
 */
static void Arm7WriteReg(unsigned char reg, unsigned long data)
{
    /* Execute LDR R[index], [R14] */
    Arm7TestExec(0xE59E0000 | ((unsigned long) reg << 12));

    /* Put the value on the bus. */
    Arm7ScanChain1(data, 0);

    /* This cycle stores the data bus value in the register, */
    Arm7ScanChain1(ARM_NOP, 0);

    /* The program counter requires two additional cycles. */
    if (reg == 15) {
        Arm7ScanChain1(ARM_NOP, 0);
        Arm7ScanChain1(ARM_NOP, 0);
    }
}

/*!
 * \brief Return index of a given register name.
 *
 * \param name Name of the register.
 *
 * \return Register index or -1 if the name is undefined.
 */
static int Arm7RegIndex(char *name)
{
    int rc;

    if (toupper(name[0]) == 'R' && isdigit(name[1])) {
        rc = atoi(name + 1);
        if (rc < 0 || rc > 15) {
            rc = -1;
        }
    } else if (stricmp(name, "SP") == 0) {
        rc = 13;
    } else if (stricmp(name, "LR") == 0) {
        rc = 14;
    } else if (stricmp(name, "PC") == 0) {
        rc = 15;
    } else if (stricmp(name, "CPSR") == 0) {
        rc = 16;
    } else {
        rc = -1;
    }
    return rc;
}

/*!
 * \brief Initialze the ICE routines.
 *
 * If in debug state, this routine will save the CPU context.
 */
void IceInit(void)
{
    ice_dsr = Arm7ScanChain2(ICE_DBG_STAT, 0);
    if ((ice_dsr & 9) == 9) {
        /* Save context. */
        Arm7SaveContext();
    }
}

/*!
 * \brief De-initialze the ICE routines.
 *
 * If in debug state, this routine will re-store the CPU context.
 */
void IceExit(void)
{
    if ((ice_dsr & 9) == 9) {
        /* Restore context. */
        Arm7WriteRegs(16, arm_regs);
    }
}

/*!
 * \brief Reset ICE routines to initial state.
 *
 * Should be called on hardware reset of the target.
 */
void IceReset(void)
{
    ice_dsr = 0;
    Arm7SelectScanChain(255);
}

/*!
 * \brief Get the device's ID code.
 *
 * \return ID code.
 */
unsigned long IceIdCode(void)
{
    unsigned long rc;

    TapInstruction(TAP_IDCODE);
    TapData((unsigned char *) &rc, NULL, 32);

    return rc;
}

/*!
 * \brief Enter debug state.
 *
 * If the CPU is running, it will be stopped and the context will
 * be saved.
 *
 * \return 0 on success, -1 otherwise.
 *
 * \todo Thumb mode is not supported yet.
 */
int IceEnterDebug(void)
{
    volatile unsigned char tmo = 0;

    if ((ice_dsr & 9) != 9) {
        /* Break on all addresses. */
        Arm7ScanChain2(ICE_WP0_ADDR | 0x20, 0x00000000);
        Arm7ScanChain2(ICE_WP0_ADDR_MASK | 0x20, 0xFFFFFFFF);

        /* Break on all data patterns. */
        Arm7ScanChain2(ICE_WP0_DATA | 0x20, 0x00000000);
        Arm7ScanChain2(ICE_WP0_DATA_MASK | 0x20, 0xFFFFFFFF);

        /* Enable watchpoint 0. */
        Arm7ScanChain2(ICE_WP0_CTRL | 0x20, 0x00000100);
        /* Detect fetch instructions. */
        Arm7ScanChain2(ICE_WP0_CTRL_MASK | 0x20, 0xFFFFFFF7);

        /* Wait until DBGACK and nMREQ are both set to 1. */
        while (((ice_dsr = Arm7ScanChain2(ICE_DBG_STAT, 0)) & 9) != 9) {
            if (++tmo == 0) {
                return -1;
            }
        }

        /* Disable watchpoints. */
        Arm7ScanChain2(ICE_WP0_CTRL | 0x20, 0x00000000);

        /* Go to Run-Test/Idle. */
        TapIdle();

        Arm7SaveContext();
    }
    return 0;
}

/*!
 * \brief Exit debug state.
 *
 * This routine will restore the context and start the CPU.
 *
 * \return 0 on success, -1 otherwise.
 */
int IceExitDebug(void)
{
    /* Store CPU status in R0 */
    Arm7WriteReg(0, arm_regs[ARM_REG_CPSR]);

    /* Execute MSR cpsr, R0 */
    Arm7TestExec(0xE12FF000);

    /* Restore context. */
    Arm7WriteRegs(16, arm_regs);

    /* Execute next instruction at MCLK. */
    Arm7ScanChain1(ARM_NOP, 1);

    /* Execute  B [PC-5] at system speed (2 for ldm + 2 instructions) */
    Arm7ScanChain1(0xEAFFFFFB, 0);
    Arm7ScanChain1(ARM_NOP, 0);

    /* Restart at system speed as soon as Run-Test Idle is reached. */
    TapInstruction(TAP_RESTART);
    TapNext(0);                 /* Go to Run-Test/Idle */
    IceReset();

    return 0;
}

/*!
 * \brief Get contents of a specified register.
 *
 * \param name Symbolic register name.
 * \param data Pointer to the variable which receives the register's contents.
 *
 * \return 0 on success, -1 otherwise.
 */
int IceCpuRegRead(char *name, unsigned long *data)
{
    int rc;

    if ((rc = Arm7RegIndex(name)) >= 0) {
        *data = arm_regs[rc];
        rc = 0;
    }
    return rc;
}

/*!
 * \brief Set contents of a specified register.
 *
 * \param name Symbolic register name.
 * \param data Value to write into the register.
 *
 * \return 0 on success, -1 otherwise.
 */
int IceCpuRegWrite(char *name, unsigned long data)
{
    int rc;

    if ((rc = Arm7RegIndex(name)) >= 0) {
        arm_regs[rc] = data;
        rc = 0;
    }
    return rc;
}

/*!
 * \brief Get contents of a specified memory location.
 *
 * \param addr Memory location to read from.
 * \param data Pointer to the variable which receives the contents.
 *
 * \return 0 on success, -1 otherwise.
 *
 * \todo In order to speed up the transfer a routine for reading multiple
 *       locations is required.
 */
int IceMemoryRead(unsigned long addr, unsigned long *data)
{
    volatile unsigned char tmo = 0;

    /* Load the memory address in r0. */
    Arm7WriteReg(0, addr);

    /* Make sure that the pipe is clean. */
    Arm7ScanChain1(ARM_NOP, 0);

    /* Execute next instruction at MCLK. */
    Arm7ScanChain1(ARM_NOP, 1);

    /* LDR R1, [R0], #4 */
    Arm7ScanChain1(0xE4901004, 0);

    Arm7ScanChain1(ARM_NOP, 0);

    /* Restart at system speed as soon as Run-Test Idle is reached. */
    TapInstruction(TAP_RESTART);
    TapNext(0);                 /* Go to Run-Test/Idle */

    /* Wait until DBGACK and nMREQ are both set to 1. */
    while ((Arm7ScanChain2(ICE_DBG_STAT, 0) & 9) != 9) {
        if (++tmo == 0) {
            return -1;
        }
    }

    /* Read the result. */
    *data = Arm7ReadReg(1);

    return 0;
}

/*!
 * \brief Set contents of a specified memory location.
 *
 * \param addr Memory location to write to.
 * \param data Value to write.
 *
 * \return 0 on success, -1 otherwise.
 *
 * \todo In order to speed up the transfer a routine for writing multiple
 *       values is required.
 */
int IceMemoryWrite(unsigned long addr, unsigned long data)
{
    volatile unsigned char tmo = 0;

    /* Load the memory address in r0. */
    Arm7WriteReg(0, addr);

    /* Load the memory contents in r1. */
    Arm7WriteReg(1, data);

    /* Make sure that the pipe is clean. */
    Arm7ScanChain1(ARM_NOP, 0);

    /* Execute next instruction at MCLK. */
    Arm7ScanChain1(ARM_NOP, 1);

    /* STR R1, [R0], #4 */
    Arm7ScanChain1(0xE4801004, 0);

    Arm7ScanChain1(ARM_NOP, 0);

    /* Restart at system speed as soon as Run-Test Idle is reached. */
    TapInstruction(TAP_RESTART);
    TapNext(0);                 /* Go to Run-Test/Idle */

    /* Wait until DBGACK and nMREQ are both set to 1. */
    while ((Arm7ScanChain2(ICE_DBG_STAT, 0) & 9) != 9) {
        if (++tmo == 0) {
            return -1;
        }
    }
    return 0;
}

/*!
 * \brief Set contents of multiple memory locations.
 *
 * \param addr Memory location to write to.
 * \param data Values to write.
 *
 * \return 0 on success, -1 otherwise.
 *
 * \todo In order to speed up the transfer a routine for writing multiple
 *       values is required.
 */
int IceMultiWrite(unsigned long addr, unsigned long *data)
{
    unsigned short tmo = 0;

    Arm7WriteRegs(12, data);


    /* Load the memory address in lr. */
    Arm7WriteReg(14, addr);

    /* Make sure that the pipe is clean. */
    Arm7ScanChain1(ARM_NOP, 0);

    /* Execute next instruction at MCLK. */
    Arm7ScanChain1(ARM_NOP, 1);

    /* Execute STMIA LR, {R0-R11}. */
    Arm7ScanChain1(0xE88E0FFF, 0);

    Arm7ScanChain1(ARM_NOP, 0);

    /* Restart at system speed as soon as Run-Test Idle is reached. */
    TapInstruction(TAP_RESTART);
    TapNext(0);                 /* Go to Run-Test/Idle */

    /* Wait until DBGACK and nMREQ are both set to 1. */
    while ((Arm7ScanChain2(ICE_DBG_STAT, 0) & 9) != 9) {
        if (++tmo == 0) {
            return -1;
        }
    }
    return 0;
}

/*!
 * \brief Send value to the debug communication channel.
 *
 * \param data Value to send to the target.
 *
 * \return 0 on success, -1 otherwise.
 */
int IceComWrite(unsigned long data)
{
    unsigned long tmo = 0x10000;

    /* Wait until target read the previously sent data. */
    while (Arm7ScanChain2(ICE_DCC_CTRL, 0) & DCC_HOST_TXBSY) {
        if (tmo-- == 0) {
            return -1;
        }
    }
    Arm7ScanChain2(ICE_DCC_DATA | 0x20, data);

    return 0;
}

/*!
 * \brief Receive value from the debug communication channel.
 *
 * \param data Pointer to the variable which will received the value
 *             sent by the target.
 *
 * \return 0 on success, -1 otherwise.
 */
int IceComRead(unsigned long *data)
{
    unsigned long tmo = 0x10000;

    /* Wait until target wrote data. */
    while ((Arm7ScanChain2(ICE_DCC_CTRL, 0) & DCC_HOST_RXRDY) == 0) {
        if (tmo-- == 0) {
            return -1;
        }
    }
    *data = Arm7ScanChain2(ICE_DCC_DATA, 0);

    return 0;
}

/*!
 * \brief Print CPU status.
 *
 * \param fp Pointer to the file stream.
 */
void IcePrintStatus(FILE * fp)
{
    int i;

    /* Print debug status. */
    if (ice_dsr & 0x10) {
        fprintf(fp, "; THUMB ");
    } else {
        fprintf(fp, "; ARM ");
    }
    if ((ice_dsr & 9) == 9) {
        fprintf(fp, "DBG\n");
    } else {
        fprintf(fp, "RUN\n");
    }

    /* Print registers. */
    for (i = 0; i < 16; i++) {
        fprintf(fp, "%s 0x%08lX\n", arm_regnames[i], arm_regs[i]);
    }
    fprintf(fp, "CPSR 0x%08lX ; ", arm_regs[ARM_REG_CPSR]);
    fputc((arm_regs[ARM_REG_CPSR] & 0x80000000) ? 'N' : '-', fp);
    fputc((arm_regs[ARM_REG_CPSR] & 0x40000000) ? 'Z' : '-', fp);
    fputc((arm_regs[ARM_REG_CPSR] & 0x20000000) ? 'C' : '-', fp);
    fputc((arm_regs[ARM_REG_CPSR] & 0x10000000) ? 'V' : '-', fp);
    fprintf(fp, "--------------------");
    fputc((arm_regs[ARM_REG_CPSR] & 0x00000080) ? 'I' : '-', fp);
    fputc((arm_regs[ARM_REG_CPSR] & 0x00000040) ? 'F' : '-', fp);
    fputc((arm_regs[ARM_REG_CPSR] & 0x00000020) ? 'T' : '-', fp);
    switch (arm_regs[ARM_REG_CPSR] & 0x1F) {
    case 0x10:
        fprintf(fp, " USER\n");
        break;
    case 0x11:
        fprintf(fp, " FIQ\n");
        break;
    case 0x12:
        fprintf(fp, " IRQ\n");
        break;
    case 0x13:
        fprintf(fp, " SUPERVISOR\n");
        break;
    case 0x17:
        fprintf(fp, " ABORT\n");
        break;
    case 0x1B:
        fprintf(fp, " UNDEFINED\n");
        break;
    case 0x1F:
        fprintf(fp, " SYSTEM\n");
        break;
    default:
        fprintf(fp, " ?????\n");
        break;
    }
}

int IceUpdate(unsigned char *image, unsigned long len)
{
    return -1;
}
