/*
 * Copyright (C) 2005 by egnite Software GmbH
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

/*!
 * \file parser.c
 * \brief JTAG-O-MAT command parser.
 *
 * \verbatim
 *
 * $Log: parser.c,v $
 * Revision 1.9  2006/03/22 19:04:22  haraldkipp
 * The LOAD command crashed if the third parameter was missing.
 * When using multiple writes, then the remaining bytes of an image had
 * been written all to the same address.
 *
 * Revision 1.8  2006/03/21 16:36:53  haraldkipp
 * Test successfully on Linux with Wiggler clone.
 *
 * Revision 1.7  2006/03/21 11:04:20  haraldkipp
 * Fixed type mismatching.
 *
 * Revision 1.6  2006/03/20 14:16:40  haraldkipp
 * Release 1.2.4.
 *
 * Revision 1.5  2006/02/05 16:44:32  haraldkipp
 * Enhanced file finder. Linux version uses DATA_DIR while special API
 * SearchPath is used on Win32.
 *
 * Revision 1.4  2006/01/09 09:03:39  haraldkipp
 * Programming adapter firmware update added.
 *
 * Revision 1.3  2005/11/25 13:40:35  haraldkipp
 * Can compile on Linux now.
 *
 * Revision 1.2  2005/11/25 09:08:22  haraldkipp
 * JOM files are searched in the PATH.
 *
 * Revision 1.1.1.1  2005/09/14 09:01:09  haraldkipp
 * Initial import.
 *
 *
 * \endverbatim
 */

#ifdef _MSC_VER
#include <windows.h>
#include <io.h>
#else
#define stricmp strcasecmp
#include <unistd.h>
#include <sys/stat.h>
#endif

#include <stdlib.h>
#include <string.h>
#include <fcntl.h>

#include "jtagomat.h"
#include "utils.h"
#include "image.h"
#include "tap.h"
#include "ice.h"
#include "parser.h"

#include "aztech.h"

#ifndef O_BINARY
#define O_BINARY  0
#endif

#define MAX_BATCH_LINE  4096

#define FLASH_GETID     0xFFFFFF01
#define FLASH_ERADEV    0xFFFFFF03
#define FLASH_ERASECT   0xFFFFFF05
#define FLASH_DATA      0xFFFFFF09

/*
 *
 */
static unsigned long Value(char *str)
{
    int base;

    if (*str == '0' && toupper(*(str + 1)) == 'X') {
        str += 2;
        base = 16;
    } else {
        base = 10;
    }
    return strtoul(str, 0, base);
}

/*
 *
 */
static unsigned long DataValue(char *str)
{
    unsigned long rc;

    if (isdigit(*str)) {
        rc = Value(str);
    } else {
        IceCpuRegRead(str, &rc);
    }
    return rc;
}

/*
 *
 */
long LoadTargetImage(char *path, unsigned long **image)
{
    int fh;
    long len = -1;

    if ((fh = open(path, O_BINARY | O_RDONLY)) != -1) {
        if ((len = filelength(fh)) > 0) {
            if ((*image = (unsigned long *)malloc(len)) != NULL) {
                if ((len = read(fh, *image, len)) <= 0) {
                    free(*image);
                    *image = NULL;
                }
            }
            close(fh);
        }
    }
    return len;
}

/*
 * BATCH file [label]
 */
int DoBatch(int argc, char **argv, int defc, char **defv, int *err)
{
    int rc = 0;
    FILE *fp;
    char *line;
    int lnum = 0;
    int cmdc;
    char **cmdv = 0;
    char **cmdp;
    char *cp;
    char *lookup;
    char pathname[255];

    if (argc < 1 || argc > 2) {
        fprintf(stderr, "Syntax error\n");
        return -1;
    }

    if (argc == 2) {
        lookup = strdup(argv[1]);
    } else {
        lookup = NULL;
    }

    strcpy(pathname, argv[0]);
    if (FindPathName(pathname, sizeof(pathname))) {
        fprintf(stderr, "Can't find %s\n", argv[0]);
        return -1;
    }
    if ((fp = fopen(pathname, "r")) == NULL) {
        fprintf(stderr, "Failed to open %s\n", pathname);
        return -1;
    }

    if ((line = (char *)malloc(MAX_BATCH_LINE)) == NULL) {
        fprintf(stderr, "No memory\n");
        fclose(fp);
        return -1;
    }

    while (fgets(line, MAX_BATCH_LINE, fp)) {
        lnum++;
        if (verbose && lookup == NULL) {
            fprintf(stderr, "%s(%d): %s", argv[0], lnum, line);
        }

        /* Chop off line terminator. */
        if ((cp = strrchr(line, '\n')) == NULL) {
            fprintf(stderr, "\nLine too long or EOL missing\n");
            rc = -1;
            break;
        }
        *cp = '\0';

        /* Chop off comments. */
        if ((cp = strrchr(line, ';')) != NULL) {
            *cp = '\0';
        }


        if ((cmdc = SplitWords(line, &cmdv)) != 0) {
            cmdp = cmdv;

            if (lookup && *lookup == 0) {
                free(lookup);
                lookup = NULL;
                continue;
            }
            /* Check for labels. */
            if (cmdp[0][strlen(cmdp[0]) - 1] == ':') {
                cmdp[0][strlen(cmdp[0]) - 1] = '\0';
                if (lookup) {
                    if (strcmp(cmdp[0], lookup)) {
                        continue;
                    }
                    free(lookup);
                    lookup = NULL;
                }
                cmdc--;
                cmdp++;
            }

            if (cmdc && lookup == NULL) {
                if (ParseCommand(cmdc, cmdp, defc, defv, &lookup)) {
                    *err = 1;
                    break;
                }
                if (lookup) {
                    if (*lookup == '\x04') {
                        break;
                    }
                    if (*lookup) {
                        fseek(fp, 0, SEEK_SET);
                        lnum = 0;
                    }
                }
            }
        }
    }

    if (lookup) {
        if (*lookup && *lookup != '\x04') {
            fprintf(stderr, "Label %s not found\n", lookup);
        }
        free(lookup);
    }
    free(line);
    fclose(fp);

    return rc;
}

/*
 * CONTINUE [addr]
 */
int DoContinue(int argc, char **argv)
{
    if (argc) {
        if (IceCpuRegWrite("PC", DataValue(argv[0]))) {
            return -1;
        }
    }
    IceExitDebug();
    return 0;
}

/*
 * DEVICE [id]
 */
int DoDevice(int argc, char **argv, int *err)
{
    unsigned long id;
    unsigned long expect;

    id = IceIdCode();
    if (verbose) {
        fprintf(stderr, "Target ID = %08lX\n", id);
    }
    if (argc) {
        expect = DataValue(argv[0]);
        if (id != expect) {
            *err = 1;
            if (verbose) {
                fprintf(stderr, "Expected 0x%08lX\n", expect);
            }
        }
    }
    return 0;
}

/*
 * FLASH base [addr [data]]
 *
 * Examples:
 * FLASH 0x10000000
 *   Retrieves the chip identifier.
 * FLASH 0x10000000 0x10000000
 *   Erases the first sector.
 * FLASH 0x10000000 0x10000000 ./image.bin
 *   Flashes the specified image.
 * FLASH 0x10000000 0x10000000 1 2 3 4
 *   Flashes the four specified 32-bit values.
 */
int DoFlash(int argc, char **argv, int *err)
{
    unsigned long *image = NULL;
    long len = 0;
    long i;
    unsigned long base;
    unsigned long offs;
    unsigned long rc;

    /* At least the base address is required. */
    if (argc < 1) {
        fprintf(stderr, "Syntax error\n");
        return -1;
    }

    /*
     * One argument only. Use it as a base address and query the ID of 
     * the flash chip at that address.
     */
    base = DataValue(argv[0]);
    argc--;
    argv++;
    if (argc == 0) {
        /* Send ID command. */
        if (IceComWrite(FLASH_GETID)) {
            return -1;
        }
        /* Send base address. */
        if (IceComWrite(base)) {
            return -1;
        }
        if (IceComRead(&rc)) {
            return -1;
        }
        if (verbose) {
            fprintf(stderr, "Flash identifier 0x%08lX\n", rc);
        }

        return 0;
    }

    /*
     * Two arguments. Erase the sector at the specified address.
     */
    offs = DataValue(argv[0]);
    argc--;
    argv++;
    if (argc == 0) {
        /* Send erase sector command. */
        if (IceComWrite(FLASH_ERASECT)) {
            return -1;
        }
        /* Send base address. */
        if (IceComWrite(base)) {
            return -1;
        }
        /* Send address offset. */
        if (IceComWrite(offs)) {
            return -1;
        }
        /* Get result. */
        if (IceComRead(&rc)) {
            return -1;
        }
        if (rc) {
            *err = 1;
        }
        return 0;
    }

// 3 parameters
    if (strchr(argv[0], '/')) {
        if ((len = LoadTargetImage(argv[0], &image)) <= 0) {
            fprintf(stderr, "Failed to read %s\n", argv[0]);
            return -1;
        }
    }

    /* Send program command. */
    if (IceComWrite(FLASH_DATA)) {
        return -1;
    }

    /* Send base address. */
    if (IceComWrite(base)) {
        return -1;
    }

    /* Send address offset. */
    if (IceComWrite(offs)) {
        return -1;
    }

    /* Send data. */
    if (image) {
        /* Send len. */
        if (IceComWrite((unsigned long) len)) {
            return -1;
        }
        len /= 4;
        for (i = 0; i < len; i++) {
            if (IceComWrite(image[i])) {
                return -1;
            }
        }
        free(image);
    } else {
        /* Send len. */
        if (IceComWrite((unsigned long) argc * 4)) {
            return -1;
        }
        while (argc--) {
            if (IceComWrite(DataValue(argv[0]))) {
                return -1;
            }
            argv++;
        }
    }

    /* Get result. */
    if (IceComRead(&rc)) {
        return -1;
    }
    if (rc) {
        *err = 1;
    }
    return 0;
}

/*
 * HALT
 */
int DoHalt(int argc, char **argv)
{
    if (IceEnterDebug()) {
        fprintf(stderr, "ERROR: Halt failed.\n");
        return -1;
    }
    if (verbose) {
        IcePrintStatus(stdout);
    }
    return 0;
}

/*
 * LOAD addr [num [data]]
 *
 * Examples:
 * LOAD 0x10000000
 * LOAD 0 128
 * LOAD 0 1 0xEA000009
 * LOAD 0 0x100 ./ram.bin
 * LOAD 0xFFE00000 32 STDOUT
 * LOAD 0 0x100 ./got.bin ./expect.bin
 */
int DoLoad(int argc, char **argv, int *err)
{
    int rc = 0;
    unsigned long addr = 0;
    char *reg = NULL;
    unsigned long val;
    unsigned long expect;
    unsigned long i;
    FILE *ofp = NULL;
    FILE *xfp = NULL;
    unsigned long num = 1;
    int pr_flg = 0;
    char pathname[255];

    if (argc < 1) {
        fprintf(stderr, "Syntax error\n");
        return -1;
    }
    if (isdigit(argv[0][0])) {
        addr = Value(argv[0]);
    } else {
        reg = argv[0];
    }
    argc--;
    argv++;
    if (argc) {
        num = (DataValue(argv[0]) + 3) / 4;
        argc--;
        argv++;
    }

    if (argc) {
        pr_flg = (stricmp(argv[0], "STDOUT") == 0);
        if (strchr(argv[0], '/') || pr_flg) {
            if (pr_flg) {
                ofp = stdout;
            } else if ((ofp = fopen(argv[0], "wb")) == NULL) {
                fprintf(stderr, "Failed to open %s\n", argv[0]);
                return -1;
            }
            argc--;
            argv++;
            if (argc && strchr(argv[0], '/')) {
                if (argv[0][0] == '.' && argv[0][1] == '/') {
                    strcpy(pathname, &argv[0][2]);
                }
                else {
                    strcpy(pathname, argv[0]);
                }
                if (FindPathName(pathname, sizeof(pathname))) {
                    fprintf(stderr, "Can't find %s\n", argv[0]);
                    return -1;
                }
                if ((xfp = fopen(pathname, "rb")) == NULL) {
                    fprintf(stderr, "Failed to open %s\n", pathname);
                    return -1;
                }
                argc--;
                argv++;
            }
        }
    }

    for (i = 0; i < num; i++) {
        if (reg) {
            rc = IceCpuRegRead(reg, &val);
        } else {
            rc = IceMemoryRead(addr, &val);
        }
        if (rc) {
            break;
        }
        if (ofp) {
            if (pr_flg) {
                if (reg) {
                    printf("%s 0x%08lX\n", reg, val);
                } else {
                    printf("0x%08lX 0x%08lX\n", addr, val);
                }
            } else {
                fwrite(&val, sizeof(val), 1, ofp);
            }
        }
        if (argc) {
            expect = DataValue(argv[0]);
            if (val != expect) {
                *err = 1;
                if (verbose) {
                    fprintf(stderr, "Got %08lX - expected 0x%08lX\n", val, expect);
                }
                break;
            }
            argc--;
            argv++;
        } else if (xfp) {
            fread(&expect, sizeof(expect), 1, xfp);
            if (val != expect) {
                *err = 1;
                if (verbose) {
                    fprintf(stderr, "0x%08lX 0x%08lX - ", addr, val);
                    fprintf(stderr, "expected 0x%08lX\n", expect);
                }
                break;
            }
        }
        addr += sizeof(val);
    }

    if (ofp && ofp != stdout) {
        fclose(ofp);
    }
    if (xfp) {
        fclose(xfp);
    }
    return rc;
}

/*
 * MODE [item[=value]]
 */
int DoMode(int argc, char **argv, int *err)
{
    return 0;
}

/*
 * NOTE Text
 */
int DoNote(int argc, char **argv)
{
    while (argc--) {
        fputs(*argv, stderr);
        argv++;
        fputc(' ', stderr);
    }
    fputc('\n', stderr);
    return 0;
}

/*
 * QUIT [result]
 *
 */
int DoQuit(int argc, char **argv)
{
    if (argc < 1) {
        return 0;
    }
    return (int) Value(argv[0]);
}

/*
 * RESET [0|1]
 */
int DoReset(int argc, char **argv)
{
    if (argc) {
        TapReset(atoi(argv[0]));
    } else {
        TapReset(1);
#ifdef _MSC_VER
        Sleep(100);
#else
        usleep(100000);
#endif
        TapReset(0);
    }
#ifdef _MSC_VER
    Sleep(250);
#else
    usleep(250000);
#endif
    TapIdle();
    IceReset();

    return 0;
}

/*
 * SAVE addr data
 *
 * Examples:
 * SAVE 0 0xEA000009
 * SAVE 0x100 ./ram.bin
 * SAVE STDIN
 */
int DoSave(int argc, char **argv, int *err)
{
    int rc = 0;
    unsigned long addr = 0;
    char *reg = NULL;
    char *line = NULL;
    char *cp;
    unsigned long val = 0;
    unsigned long multi[12];
    IMAGEFILE *ifp = NULL;
    int ac;
    char **av = NULL;

    /* At least one argument is required. */
    if (argc < 1) {
        fprintf(stderr, "Syntax error\n");
        return -1;
    }

    /* First argument may be an address value. */
    if (isdigit(argv[0][0])) {
        addr = Value(argv[0]);
    } else {
        /* If first argument is 'STDIN', then we expect address/data pairs on stdin. */
        if (stricmp(argv[0], "STDIN") == 0) {
            line = (char *)malloc(256);
        } 
        /* If no digit and not STDIN, then we assume a register name. */
        else {
            reg = argv[0];
        }
    }
    argc--;
    argv++;

    /* Second argument may specify the name of an image file. */
    if (argc && strchr(argv[0], '/')) {
        if ((ifp = ImageOpen(argv[0])) == NULL) {
            fprintf(stderr, "Failed to open %s\n", argv[0]);
            return -1;
        }
        argc--;
        argv++;
    }

    while (rc == 0) {
        /* When reading from a file, we use JTAG multi-write to speed up the transfer. */
        if (ifp) {
            size_t mcnt;
            size_t lcnt;

            if ((mcnt = ImageRead(ifp, &addr, (unsigned char *)multi, 12 * sizeof(multi[0]))) == 0) {
                ImageClose(ifp);
                ifp = NULL;
                continue;
            }
            mcnt /= sizeof(multi[0]);
            if (mcnt == 12) {
                rc = IceMultiWrite(addr, multi);
                addr += mcnt * sizeof(multi[0]);
            }
            else {
                for (lcnt = 0; lcnt < mcnt; lcnt++) {
                    if ((rc = IceMemoryWrite(addr, multi[lcnt])) != 0) {
                        break;
                    }
                    addr += sizeof(multi[0]);
                }
            }
            continue;
        } else if (line) {
            if (fgets(line, 255, stdin) == NULL) {
                break;
            }
            if ((cp = strchr(line, ';')) != NULL) {
                *cp = 0;
            } else if ((cp = strchr(line, '\n')) != NULL) {
                *cp = 0;
            }
            if ((ac = SplitWords(line, &av)) == 2) {
                if (isdigit(av[0][0])) {
                    addr = Value(av[0]);
                    reg = NULL;
                } else {
                    reg = av[0];
                }
                val = Value(av[1]);
            }
        } else if (argc) {
            val = DataValue(argv[0]);
            argc--;
            argv++;
        } else {
            break;
        }
        if (reg) {
            rc = IceCpuRegWrite(reg, val);
        } else {
            rc = IceMemoryWrite(addr, val);
        }
        addr += sizeof(val);
    }

    if (line) {
        free(line);
    }
    if (av) {
        for (ac = 0; av[ac]; ac++) {
            free(av[ac]);
        }
        free(av);
    }
    if (ifp) {
        ImageClose(ifp);
    }
    return rc;
}

/*
 * UPDATE image
 */
int DoUpdate(int argc, char **argv)
{
    unsigned long *image = NULL;
    int rc = -1;
    long len;

    if (argc < 1) {
        fprintf(stderr, "Syntax error\n");
        return rc;
    }
    if ((len = LoadTargetImage(argv[0], &image)) <= 0) {
        fprintf(stderr, "Failed to read %s\n", argv[0]);
        return rc;
    }
    if (image) {
        rc = IceUpdate((unsigned char *)image, (unsigned long)len);
        free(image);
#ifdef _MSC_VER
        Sleep(1000);
#else
        usleep(1000000);
#endif
    }
    return rc;
}

/*
 * WAIT [time]
 *
 */
int DoWait(int argc, char **argv)
{
    unsigned long ms;

    if (argc < 1) {
        ms = 1000;
    } else {
        ms = Value(argv[0]);
    }
#ifdef _MSC_VER
    Sleep(ms);
#else
    usleep(ms * 1000);
#endif

    return 0;
}

/*!
 * \brief Recursively expand all macros.
 */
static int ExpandMacros(int argc, char **argv, int defc, char **defv)
{
    int rc = 0;
    int xc;
    int xl = 0;
    char *argp;
    int argl;
    int i;
    int j;

    do {
        /* Limit the number of iterations. */
        if (xl++ > 32) {
            rc = -1;
            break;
        }

        /* Loop through all arguments. */
        xc = 0;
        for (i = 0; i < argc; i++) {
            if (argv[i][0] == '$') {
                argp = &argv[i][1];
                argl = strlen(argp);
                for (j = 0; j < defc; j++) {
                    if (strncmp(argp, defv[j], argl) == 0 && defv[j][argl] == '=') {
                        /* Replace the macro. */
                        xc++;
                        free(argv[i]);
                        argv[i] = strdup(&defv[j][argl + 1]);
                        break;
                    }
                }

                /* No definition found, try the shell environment. */
                if (j >= defc) {
                    if ((argp = getenv(argp)) == NULL) {
                        fprintf(stderr, "Unknown macro %s\n", argv[i]);
                        rc = -1;
                        break;
                    }
                    /* Replace the macro. */
                    xc++;
                    free(argv[i]);
                    argv[i] = strdup(argp);
                }
            }
        }
        /* Loop until all macros had been replaced. */
    } while (xc);

    return rc;
}

/*
 * BATCH file [label]
 *
 * RESET
 * HALT
 * CONTINUE [addr]
 * WAIT
 * QUIT
 *
 * DEVICE [id]
 *
 * LOAD addr [num [data]]
 * SAVE addr [data]
 * FLASH [addr [data]]
 *
 * MODE [item[=value]]
 * JUMP label
 * ONERR label
 * ; Comment
 * label:
 *
 * GET/PUT reserved for comms interface.
 * ICE reserved for EmbeddedICE-RT logic.
 */
int ParseCommand(int argc, char **argv, int defc, char **defv, char **label)
{
    static int err = 0;
    int rc = -1;
    size_t len;

    /* Expand macros in batch files only (label not NULL). */
    if (argc && (label == NULL || ExpandMacros(argc, argv, defc, defv) == 0)) {
        strupr(argv[0]);
        len = strlen(argv[0]);
        if (strncmp(argv[0], "BATCH", len) == 0) {
            rc = DoBatch(argc - 1, &argv[1], defc, defv, &err);
        }

        else if (strncmp(argv[0], "UPDATE", len) == 0) {
            rc = DoUpdate(argc - 1, &argv[1]);
        }
        else if (strncmp(argv[0], "RESET", len) == 0) {
            rc = DoReset(argc - 1, &argv[1]);
        } else if (strncmp(argv[0], "HALT", len) == 0) {
            rc = DoHalt(argc - 1, &argv[1]);
        } else if (strncmp(argv[0], "CONTINUE", len) == 0) {
            rc = DoContinue(argc - 1, &argv[1]);
        } else if (strncmp(argv[0], "WAIT", len) == 0) {
            rc = DoWait(argc - 1, &argv[1]);
        } else if (label && strncmp(argv[0], "QUIT", len) == 0) {
            rc = DoQuit(argc - 1, &argv[1]);
            *label = strdup("\x04");
        } else if (strncmp(argv[0], "NOTE", len) == 0) {
            rc = DoNote(argc - 1, &argv[1]);
        }

        else if (strncmp(argv[0], "DEVICE", len) == 0) {
            rc = DoDevice(argc - 1, &argv[1], &err);
        }

        else if (strncmp(argv[0], "LOAD", len) == 0) {
            rc = DoLoad(argc - 1, &argv[1], &err);
        } else if (strncmp(argv[0], "SAVE", len) == 0) {
            rc = DoSave(argc - 1, &argv[1], &err);
        } else if (strncmp(argv[0], "FLASH", len) == 0) {
            rc = DoFlash(argc - 1, &argv[1], &err);
        }

        else if (strncmp(argv[0], "MODE", len) == 0) {
            rc = DoMode(argc - 1, &argv[1], &err);
        } else if (label && strncmp(argv[0], "JUMP", len) == 0) {
            if (argc > 1) {
                *label = strdup(argv[1]);
            } else {
                *label = strdup("");
            }
            rc = 0;
        } else if (label && strncmp(argv[0], "ONERR", len) == 0) {
            if (err) {
                err = 0;
                if (argc > 1) {
                    *label = strdup(argv[1]);
                } else {
                    *label = strdup("");
                }
            }
            rc = 0;
        } else if (strncmp(argv[0], "VEGAFLASH", len) == 0) {
	            rc = DoVegaFlash(argc - 1, &argv[1], &err);
        } else {
            fprintf(stderr, "Unknown command %s\n", argv[0]);
        }
    }
    return rc;
}
