/*
 * Copyright (C) 2005 by egnite Software GmbH
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

/*!
 * \file image.c
 * \brief Image file I/O routines.
 *
 * \verbatim
 *
 * $Log: image.c,v $
 * Revision 1.2  2006/03/21 11:04:20  haraldkipp
 * Fixed type mismatching.
 *
 * Revision 1.1  2006/03/20 14:16:40  haraldkipp
 * Release 1.2.4.
 *
 *
 * \endverbatim
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "utils.h"
#include "image.h"

#define RECTYP_DAT  0
#define RECTYP_EOF  1
#define RECTYP_XSEG 2
#define RECTYP_SSEG 3
#define RECTYP_XLIN 4
#define RECTYP_SLIN 5
#define RECTYP_BAD  -1

/*!
 * \brief Read value from hex file.
 *
 * \param fp     File to read from.
 */
static int ReadHexByte(FILE *fp)
{
    int val = 0;
    int ch;
    int i;

    for (i = 0; i < 2; i++) {
        ch = fgetc(fp);
        if (ch >= '0' && ch <= '9') {
            val <<= 4;
            val += ch - '0';
        }
        else if (ch >= 'A' && ch <= 'F') {
            val <<= 4;
            val += ch - 'A' + 10;
        }
        else {
            return -1;
        }
    }
    return val;
}

static int ReadIntelHexLine(FILE *fp, int *ldoffs, int *reclen, unsigned char *data)
{
    int typ;
    int cks;
    int val;
    int len;

    /* First character must be colon. Skip EOL characters. */
    while ((val = fgetc(fp)) != ':') {
        if (val != 10 && val != 13) {
            return RECTYP_BAD;
        }
    }

    /* First byte specifies the record length. */
    if ((len = ReadHexByte(fp)) < 0) {
        return RECTYP_BAD;
    }
    cks = len;
    if (reclen) {
        *reclen = len;
    }

    /* Next two bytes specify an offset value. */
    if ((val = ReadHexByte(fp)) < 0) {
        return RECTYP_BAD;
    }
    cks += val;
    if (ldoffs) {
        *ldoffs = val << 8;
    }
    if ((val = ReadHexByte(fp)) < 0) {
        return RECTYP_BAD;
    }
    cks += val;
    if (ldoffs) {
        *ldoffs += val;
    }

    /* Next byte specifies the record type. */
    if ((typ = ReadHexByte(fp)) < 0) {
        return RECTYP_BAD;
    }
    cks += typ;

    /* Remaining digits minus two specify the data. */
    while (len--) {
        if ((val = ReadHexByte(fp)) < 0) {
            return RECTYP_BAD;
        }
        if (data) {
            *data++ = (unsigned char)val;
        }
        cks += val;
    }

    /* Last two digits are the checksum. */
    cks = -cks & 0xFF;
    if (cks != ReadHexByte(fp)) {
        return RECTYP_BAD;
    }
    return typ;
}

/*!
 * \brief Open an existing image file.
 *
 * \param filename Pathname of the file to open.
 *
 * \return Pointer to an IMAGEFILE structure on success or NULL in case
 *         of an error.
 */
IMAGEFILE *ImageOpen(char *filename)
{
    IMAGEFILE *imf;

    if ((imf = (IMAGEFILE *)calloc(1, sizeof(IMAGEFILE))) == NULL) {
        return NULL;
    }

    if (filename[0] == '.' && filename[1] == '/') {
        strcpy(imf->imf_pathname, filename + 2);
    }
    else {
        strcpy(imf->imf_pathname, filename);
    }
    if (FindPathName(imf->imf_pathname, sizeof(imf->imf_pathname))) {
        free(imf);
        return NULL;
    }

    if ((imf->imf_fp = fopen(imf->imf_pathname, "rb")) == NULL) {
        free(imf);
        return NULL;
    }

    if (ReadIntelHexLine(imf->imf_fp, NULL, NULL, NULL) != RECTYP_BAD) {
        imf->imf_type = IMAGETYPE_IHEX;
    }
    else {
        imf->imf_type = IMAGETYPE_BIN;
    }
    fseek(imf->imf_fp, 0, SEEK_SET);

    return imf;
}

/*!
 * \brief Create an image file.
 *
 * Not yet implemented.
 *
 * \param filename Pathname of the file to create.
 * \param type     Type of the new file, either IMAGETYPE_BIN or 
 *                 IMAGETYPE_IHEX.
 *
 * \return Always NULL.
 */
IMAGEFILE *ImageCreate(char *filename, int type)
{
    return NULL;
}

/*!
 * \brief Read from a previously opened image file.
 */
int ImageRead(IMAGEFILE *imf, unsigned long *addr, unsigned char *data, size_t siz)
{
    int rc = 0;
    size_t cnt;
    unsigned long ldpos = *addr;

    while (siz) {
        if (imf->imf_type == IMAGETYPE_IHEX) {
            int reclen;
            int ldoffs;
            int rectyp;

            while (!imf->imf_end && imf->imf_bcnt == 0) {
                imf->imf_bptr = imf->imf_buff;
                rectyp = ReadIntelHexLine(imf->imf_fp, &ldoffs, &reclen, imf->imf_buff);
                if (rectyp == RECTYP_DAT) {
                    if (ldpos != imf->imf_seg + ldoffs) {
                        /* Address gap. */
                        if (rc) {
                            break;
                        }
                        ldpos = imf->imf_seg + ldoffs;
                        *addr = ldpos;
                    }
                    imf->imf_bcnt = reclen;
                }
                else if (rectyp == RECTYP_XSEG) {
                    imf->imf_seg = imf->imf_buff[0];
                    imf->imf_seg <<= 8;
                    imf->imf_seg += imf->imf_buff[1];
                    imf->imf_seg <<= 4;
                }
                else if (rectyp == RECTYP_BAD || rectyp == RECTYP_EOF) {
                    imf->imf_end = 1;
                }
            }
        }
        else if (!imf->imf_end && imf->imf_bcnt == 0) {
            imf->imf_bptr = imf->imf_buff;
            imf->imf_bcnt = fread(imf->imf_buff, 1, sizeof(imf->imf_buff), imf->imf_fp);
        }
        if (imf->imf_bcnt == 0) {
            break;
        }
        if (imf->imf_bcnt <= siz) {
            cnt = imf->imf_bcnt;
        }
        else {
            cnt = siz;
        }
        memcpy(data, imf->imf_bptr, cnt);
        data += cnt;
        imf->imf_bptr += cnt;
        imf->imf_bcnt -= cnt;
        ldpos += cnt;
        siz -= cnt;
        rc += cnt;
    }
    return rc;
}

int ImageClose(IMAGEFILE *imf)
{
    int rc = -1;

    if (imf) {
        if (imf->imf_fp) {
            rc = fclose(imf->imf_fp);
        }
        free(imf);
    }
    return rc;
}

