/*
 * Copyright (C) 2005 by egnite Software GmbH
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

/*!
 * \file parport.c
 * \brief Hardware dependant part of the physical JTAG layer.
 *
 * This file is used for the Windows PC parallel port and requires the
 * installation of the giveio.sys driver.
 *
 * \verbatim
 *
 * $Log: parport.c,v $
 * Revision 1.3  2006/03/21 16:36:53  haraldkipp
 * Test successfully on Linux with Wiggler clone.
 *
 * Revision 1.2  2006/03/20 14:14:39  haraldkipp
 * Added Linux and BSD support for parallel port access.
 *
 * Revision 1.1.1.1  2005/09/14 09:01:09  haraldkipp
 * Initial import.
 *
 *
 * \endverbatim
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_SYS_IO_H
#include <sys/io.h>
#endif

#if defined(_WIN32)

#include <conio.h>
#include <windows.h>
#define DIRECT_PORT_ACCESS
#define PORTDRIVER  "\\\\.\\giveio"

static HANDLE fdpar;
static unsigned short port_base = 0x378;

#elif defined(__FreeBSD__)
/* BSD is untested and may not work. */

#include <sys/dev/ppbus/ppi.h>

#define stricmp strcasecmp

#define PPRSTATUS       PPIGSTATUS
#define PPWDATA         PPISDATA

static int fdpar;
static unsigned int port_base;

#elif defined(__linux__)

#include <fcntl.h>
#include <sys/ioctl.h>
#include <linux/parport.h>
#include <linux/ppdev.h>
#include <errno.h>

#define stricmp strcasecmp

static int fdpar;
static unsigned int port_base;
extern FILE* logfile;
#endif /* __linux__ */


#include "jtagomat.h"
#include "tap.h"
#include "aztech.h"

/* Wiggler uses these pins. */
#define PORT_RST_MASK   0x01    /* Pin 2 */
#define PORT_TMS_MASK   0x02    /* Pin 3 */
#define PORT_TCK_MASK   0x04    /* Pin 4 */
#define PORT_TDI_MASK   0x08    /* Pin 5 */
#define PORT_TDO_MASK   0x80    /* Pin 11 */

static int port_shadow = PORT_RST_MASK;


/*!
 * \brief Set TDI bit.
 *
 * \param s Specifies wether TDI should be set (1) or cleared (0).
 */
void TapTdi(int s)
{
    if (s) {
        port_shadow |= PORT_TDI_MASK;
		LOG("1");
    } else {
        port_shadow &= ~PORT_TDI_MASK;
		LOG("0");
    }
}

/*!
 * \brief Get TDO bit.
 *
 * \return 1 if TDO is active, 0 otherwise.
 */
int TapTdo(void)
{
#if defined(__linux__) || defined(__FreeBSD__)

    unsigned char temp;

#ifdef DIRECT_PORT_ACCESS
    __asm__ __volatile__ (      /* */
        "inb    %w1, %0"        /* */
        :"=a" (temp)            /* */
        :"Nd" (port_base + 1)   /* */
    );
#else /* !DIRECT_PORT_ACCESS */
    ioctl(fdpar, PPRSTATUS, &temp);
#endif /* !DIRECT_PORT_ACCESS */

    if (temp & PORT_TDO_MASK) {
        return 0;
    }

#else /* !__linux__ && !__FreeBSD__ */

    if (_inp((unsigned short) (port_base + 1)) & PORT_TDO_MASK) {
        return 0;
    }

#endif /* !__linux__ */
    return 1;
}

/*!
 * \brief Go to next TAP state.
 *
 * \param s Specifies wether TMS should be set (1) or cleared (0).
 */
void TapNext(int s)
{
    if (s) {
        port_shadow |= PORT_TMS_MASK;
    } else {
        port_shadow &= ~PORT_TMS_MASK;
    }

#if defined(__linux__) || defined(__FreeBSD__)
    {
        unsigned char temp = (unsigned char)port_shadow;

#ifdef DIRECT_PORT_ACCESS
        __asm__ __volatile__ (  /* */
            "outb   %b0, %w1"   /* */
            :                   /* */
            :"a" (temp)         /* */
            , "Nd" (port_base)  /* */
        );
#else /* !DIRECT_PORT_ACCESS */
        ioctl(fdpar, PPWDATA, &temp);
#endif /* !DIRECT_PORT_ACCESS */
        temp |= PORT_TCK_MASK;
#ifdef DIRECT_PORT_ACCESS
        __asm__ __volatile__ (  /* */
            "outb   %b0, %w1"   /* */
            :                   /* */
            :"a" (temp)         /* */
            , "Nd" (port_base)  /* */
        );
#else /* !DIRECT_PORT_ACCESS */
        ioctl(fdpar, PPWDATA, &temp);
#endif /* !DIRECT_PORT_ACCESS */
    }

#else /* !__linux__ && !__FreeBSD__ */

    /* Rising edge at TCK. */
    _outp(port_base, port_shadow);
    _outp(port_base, port_shadow | PORT_TCK_MASK);

#endif /* !__linux__ && !__FreeBSD__ */
}

/*!
 * \brief Reset TAP controller.
 *
 * \param s Specifies wether TRST should be set (1) or cleared (0).
 */
void TapReset(int s)
{
#if defined(__linux__) || defined(__FreeBSD__)

    unsigned char temp;

    if (s) {
        temp = 0;
    } else {
        temp = PORT_RST_MASK;
    }
#ifdef DIRECT_PORT_ACCESS
        __asm__ __volatile__ (  /* */
            "outb   %b0, %w1"   /* */
            :                   /* */
            :"a" (temp)         /* */
            , "Nd" (port_base)  /* */
        );
#else /* !DIRECT_PORT_ACCESS */
    ioctl(fdpar, PPWDATA, &temp);
#endif /* !DIRECT_PORT_ACCESS */

#else /* !__linux__ && !__FreeBSD__ */

    if (s) {
        _outp(port_base, 0);
    } else {
        _outp(port_base, PORT_RST_MASK);
    }

#endif /* !__linux__ && !__FreeBSD__ */
}

/*!
 * \brief Initialize TAP controller.
 *
 * \param devname Name of the device, eg. lpt1, lpt2 or lpt3 on
 *                Windows or parport0 or parport1 on Linux. May be
 *                NULL or point to an empty string to stick with
 *                the default.
 *
 * \return 0 on success, -1 otherwise.
 *
 * \todo Implementing the Linux version.
 */
int TapInit(char *devname)
{

    if (devname == NULL || *devname == '\0') {
#if defined(_WIN32)
        devname = "LPT1";
#elif defined(__linux__)
        devname = "/dev/parport0";
#elif defined(__FreeBSD__)
        devname = "/dev/lpt0";
#endif
    }

#ifdef DIRECT_PORT_ACCESS
    if (stricmp(devname, "lpt1") == 0 ||        /* Win32 name. */
        stricmp(devname, "/dev/parport0") == 0) {    /* Linux name. */
        port_base = 0x378;
    } else if (stricmp(devname, "lpt2") == 0 || /* Win32 name. */
        stricmp(devname, "/def/parport1") == 0) {    /* Linux name. */
        port_base = 0x278;
    } else if (stricmp(devname, "lpt3") == 0 || /* Win32 name. */
        stricmp(devname, "/dev/parport2") == 0) {    /* Linux name. */
        port_base = 0x3BC;
    } else {
        fprintf(stderr, "Unknown device %s for direct access\n", devname);
        return -1;
    }
#endif

    if (verbose) {
        fprintf(stderr, "Wiggler at %s", devname);
#ifdef DIRECT_PORT_ACCESS
        fprintf(stderr, " - Port 0x%X", port_base);
#endif /* DIRECT_PORT_ACCESS */
        fputc('\n', stderr);
    }

#if defined(__linux__) || defined(__FreeBSD__)

#ifdef DIRECT_PORT_ACCESS
    if (ioperm(port_base, 3, 1)) {
        return -1;
    }

#else /* !DIRECT_PORT_ACCESS */

    if ((fdpar = open(devname, O_RDWR)) == -1) {
        fprintf(stderr, "Opening %s: %s\n", devname, strerror(errno));
        return -1;
    }
#if defined(__linux__)
    if (ioctl(fdpar, PPCLAIM)) {
        fprintf(stderr, "Claiming %s: %s\n", devname, strerror(errno));
        close(fdpar);
        return -1;
    }
#endif /* __linux__ */

#endif /* !DIRECT_PORT_ACCESS */

#else /* !__linux__ && !__FreeBSD__ */

    fdpar = CreateFile(PORTDRIVER, GENERIC_READ, 0, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
    if (fdpar == INVALID_HANDLE_VALUE) {
        return -1;
    }
    CloseHandle(fdpar);


#endif /* !__linux__ && !__FreeBSD__ */
    TapIdle();

    return 0;
}

/*!
 * \brief De-initialize the TAP controller.
 */
void TapExit(void)
{
    TapReset(0);
#if defined(__linux__) || defined(__FreeBSD__)

#ifndef DIRECT_PORT_ACCESS
    if (fdpar != -1) {
#if defined(__linux__)
        ioctl(fdpar, PPRELEASE);
#endif /* __linux__ */
        close(fdpar);
    }
#endif /* DIRECT_PORT_ACCESS */

#endif /* __linux__ || __FreeBSD__ */
}
