/*
 * Copyright (C) 2005 by egnite Software GmbH
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

/*!
 * \file tap.c
 * \brief Hardware independant part of the physical JTAG layer.
 *
 * \verbatim
 *
 * $Log: tap.c,v $
 * Revision 1.1.1.1  2005/09/14 09:01:09  haraldkipp
 * Initial import.
 *
 *
 * \endverbatim
 */

#include <stdlib.h>
#include "aztech.h"

#include "tap.h"

// this function will scan in wbuf in the following order:
// 1. get the first byte from wbuf
// 2. send the byte from lowest position.
// 3. get the next byte
// so in order to send "10001000 01101111"
// the data given should be as followed:
// "00010001 11110110" (reverse the sequence of each byte)

// for reading: 
// while (len) {while (len && mask) {mask <<= 1;} *rbuf++ = val; } 
// shows that one circle is 8 bytes, and the older is from lower to higher for each byte
// for bytes, the higher bytes goes first. so the bit order goes like
// 87654321 (16)(15)(14)(13)(12)(11)(10)9
static void TapRegister(unsigned char *rbuf, const unsigned char *wbuf, int len)
{
	int count = 0;
    while (len) {
        unsigned char mask = 1;
        unsigned char val = 0;

        if (wbuf) {
            val = *wbuf++;
        }
        while (len && mask) {
			count ++;
            TapTdi(val & mask);
			if(count%8 == 0)
				LOG(" ");
 		//will keep in the "shift IR" Tap state if (TMS=0)
            TapNext(--len == 0);
            if (TapTdo()) {
                val |= mask;
            } else {
                val &= ~mask;
            }
            mask <<= 1;
        }
        if (rbuf) {
            *rbuf++ = val;
        }
    }
	LOG("\n");
}

/*!
 * \brief Read and write data register.
 */
void TapData(unsigned char *rbuf, const unsigned char *wbuf, int len)
{
	LOG("scan in data:\n");
    /* Run-Test/Idle, Update-DR or Update-IR state. */
    TapNext(1);                 /* -> Select-DR-Scan */
    TapNext(0);                 /* -> Capture-DR */
    TapNext(0);                 /* -> Shift-DR */

    TapRegister(rbuf, wbuf, len);

    TapNext(1);                 /* Exit1-DR -> Update-DR */
}

/*!
 * \brief Store instruction code in instruction register.
 */
void TapInstruction(unsigned long code)
{
	LOG("scan in instruction:  ");

    /* Run-Test/Idle, Update-DR or Update-IR state. */
    TapNext(1);                 /* -> Select-DR-Scan */
    TapNext(1);                 /* -> Select-IR-Scan */
    TapNext(0);                 /* -> Capture-IR */
    TapNext(0);                 /* -> Shift-IR */

    TapRegister(NULL, (unsigned char *) &code, 4);
	 //Shift - IR  ->  Exit1-IR is done in TapRegister.
	 
    TapNext(1);                 /* Exit1-IR -> Update-IR */
}

/*!
 * \brief Go to Run-Test/Idle state.
 */
void TapIdle(void)
{
    int i;

    /* Any -> Test-Logic-Reset */
    for (i = 0; i < 5; i++) {
        TapNext(1);
    }
    TapNext(0);                 /* -> Run-Test/Idle */
}
