/*
 * Copyright (C) 2005-2006 by egnite Software GmbH
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

/*!
 * \file utils.c
 * \brief Utility routines.
 *
 * \verbatim
 *
 * $Log: utils.c,v $
 * Revision 1.3  2006/03/21 16:36:53  haraldkipp
 * Test successfully on Linux with Wiggler clone.
 *
 * Revision 1.2  2006/03/21 11:04:20  haraldkipp
 * Fixed type mismatching.
 *
 * Revision 1.1  2006/03/20 14:16:40  haraldkipp
 * Release 1.2.4.
 *
 *
 * \endverbatim
 */

#ifdef _MSC_VER
#include <windows.h>
#include <io.h>
#else
#include <string.h>
#include <stdlib.h>
#define stricmp strcasecmp
#include <unistd.h>
#include <sys/stat.h>
#endif

#include "utils.h"

#ifndef _MSC_VER
char *strupr(char *str)
{
    char *cp = str;
    while(*cp) {
        *cp = toupper(*cp);
        cp++;
    }
    return str;
}

int filelength(int fd)
{
    struct stat s;
    if (fstat(fd, &s) < 0) {
      return 0;
    }
    return s.st_size;
}
#endif

int SplitWords(char *line, char ***array)
{
    char **argv;
    char *wp;
    char *lp;
    int i;
    int j;
    int n;
    int maxlen;

    /* Free a previously allocated array. */
    argv = *array;
    if (argv) {
        for (i = 0; argv[i]; i++) {
            free(argv[i]);
        }
        free(argv);
    }

    /* Determine the number of words and the max. word length. */
    n = maxlen = 0;
    for (lp = line; *lp;) {
        while (*lp == ' ' || *lp == '\t') {
            ++lp;
        }
        if (*lp) {
            wp = lp;
            while (*lp && *lp != ' ' && *lp != '\t') {
                ++lp;
            }
            ++n;
            if ((i = lp - wp) > maxlen) {
                maxlen = i;
            }
        }
    }

    /*
       * Allocate enough for that many words plus 1 for null
     */
    argv = (char **)malloc((size_t) (n + 1) * sizeof(char *));

    /*
       * Allocate enough to fit the longest word
     */
    wp = (char *)malloc((size_t) maxlen + 1);

    j = i = 0;
    for (;;) {                  /* Now build the list of words */
        while (*line == ' ' || *line == '\t') {
            line++;
        }
        if (*line == '\0') {
            break;
        }

        i = 0;
        for (; *line != ' ' && *line != '\t' && *line != '\0'; ++line) {
            wp[i++] = *line;
        }
        wp[i] = '\0';
        argv[j++] = strdup(wp);
        if (*line == '\0') {
            break;
        }
    }
    argv[j] = NULL;             /* remember null at end of list */
    *array = argv;
    free(wp);

    return j;
}

int FindPathName(char *path, size_t maxlen)
{
    char *cp;
    char *ori;
    char *rel;

    /*
     * Try the original path first.
     */
    if (access(path, 0) == 0) {
        return 0;
    }

    /*
     * Give up if this is an absolute path.
     */
    if (*path == '/' || *path == '\\' || *(path + 1) == ':') {
        return -1;
    }

    /*
     * Create a copy with UNIX path delimiters and skip leading
     * dot slash.
     */
    if ((ori = strdup(path)) == NULL) {
        return -1;
    }
    while ((cp = strchr(ori, '\\')) != NULL) {
        *cp = '/';
    }
    rel = ori;
    if (*rel == '.' && *(rel + 1) == '/') {
        rel += 2;
    }

#ifdef DATA_DIR
    /*
     * When build with autotools, we should have a data directory
     * defined, which is typically /usr/local/etc.
     */
    {
        char *ddp;
        size_t len = strlen(rel) + sizeof(DATA_DIR) + 1;

        if (len >= maxlen || (ddp = (char *)malloc(len)) == NULL) {
            return -1;
        }
        strcpy(ddp, DATA_DIR);
        if (*ddp && ddp[strlen(DATA_DIR) - 1] != '/') {
            strcat(ddp, "/");
        }
        strcat(ddp, rel);
        if (access(ddp, 0) == 0) {
            strcpy(path, ddp);
            free(ddp);
            free(ori);
            return 0;
        }
        free(ddp);
    }
#endif

#ifdef _WIN32

    /*
     * The Windows installation stores the scripts in the same
     * directory as our executable. The SearchPath() API should
     * find it.
     */
    if (SearchPath(NULL, rel, NULL, maxlen, path, &cp)) {
        free(ori);
        return access(path, 0);
    }

#else /* ! _WIN32 */

    /*
     * On other systems search the PATH environment.
     */
    {
        char *env = getenv("PATH");

        if (env == NULL) {
            free(ori);
            return -1;
        }

        while (*env) {
            cp = path;
            while (*env) {
                if (*env == ';' || *env == ':') {
                    env++;
                    break;
                }
                if (*env == '\\') {
                    *cp++ = '/';
                }
                else {
                    *cp++ = *env;
                }
                env++;
            }
            *cp++ = '/';
            strcpy(cp, ori);
            if (access(path, 0) == 0) {
                free(ori);
                return 0;
            }
        }
    }

#endif /* ! _WIN32 */

    free(ori);
    return -1;
}

